<?php
if ( ! defined( 'ABSPATH' ) ) {

	exit; // Exit if accessed directly.
}

/**
 * Main Logistik Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */

final class Logistik_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */

	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';


	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */

	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated

		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version

		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}


		// Add Plugin actions

		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );


        // Register widget scripts

		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);


		// Specific Register widget scripts

		// add_action( 'elementor/frontend/after_register_scripts', [ $this, 'logistik_regsiter_widget_scripts' ] );
		// add_action( 'elementor/frontend/before_register_scripts', [ $this, 'logistik_regsiter_widget_scripts' ] );


        // category register

		add_action( 'elementor/elements/categories_registered',[ $this, 'logistik_elementor_widget_categories' ] );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'logistik' ),
			'<strong>' . esc_html__( 'Logistik Core', 'logistik' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'logistik' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */

			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'logistik' ),
			'<strong>' . esc_html__( 'Logistik Core', 'logistik' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'logistik' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(

			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'logistik' ),
			'<strong>' . esc_html__( 'Logistik Core', 'logistik' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'logistik' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */

	public function init_widgets() {

		$widget_register = \Elementor\Plugin::instance()->widgets_manager;

		// Header Elements
		require_once( LOGISTIK_ADDONS . '/header/header.php' );

		// Include Widget files
		foreach($this->Logistik_Include_File() as $widget_file_name){
			require_once( LOGISTIK_ADDONS . '/widgets/logistik-'."$widget_file_name".'.php' );
		}

		// Header Widget Register
		$widget_register->register ( new \Logistik_Header() );
		
		// Register widget
		$widget_register->register ( new \Logistik_Section_Title() );
		$widget_register->register ( new \Logistik_Button() );
		$widget_register->register ( new \Logistik_Blog() );
		$widget_register->register ( new \Logistik_Banner() );
		$widget_register->register ( new \logistik_Video() );
		$widget_register->register ( new \Logistik_Team() );
		$widget_register->register ( new \Logistik_Team_info() );
		$widget_register->register ( new \Logistik_Arrows() );
		$widget_register->register ( new \Logistik_Testimonial() );
		$widget_register->register ( new \Logistik_Service() );
		$widget_register->register ( new \logistik_Portfolio() );
		$widget_register->register ( new \logistik_Portfolio_Info() );
		$widget_register->register ( new \Logistik_Image() );
		$widget_register->register ( new \Logistik_Animated_Shape() );
		$widget_register->register ( new \Logistik_Brand_Logo() );
		$widget_register->register ( new \Logistik_Counterup() );
		$widget_register->register ( new \Logistik_Social() );
		$widget_register->register ( new \Logistik_Contact_Info() );
		$widget_register->register ( new \Logistik_Contact_Form() );
		$widget_register->register ( new \Logistik_Faq() );
		$widget_register->register ( new \Logistik_Tab_Builder() );
		$widget_register->register ( new \Logistik_Gallery() );
		$widget_register->register ( new \Logistik_Info_Box() );
		$widget_register->register ( new \logistik_Skill() );
		$widget_register->register ( new \logistik_Newsletter() );
		$widget_register->register ( new \Logistik_About_Info() );
		$widget_register->register ( new \Logistik_Cta() );
		$widget_register->register ( new \logistik_Step() );
		$widget_register->register ( new \logistik_Request_Quote() );
		$widget_register->register ( new \logistik_Features() );
	}

	public function Logistik_Include_File(){
		return ['section-title', 'button', 'banner', 'blog', 'video', 'team', 'team-info', 'arrows', 'testimonial', 'service', 'portfolio', 'portfolio-info', 'image', 'animated-shape', 'brand-logo', 'counterup', 'social', 'contact-info', 'contact-form', 'faq', 'tab-builder', 'gallery', 'info-box', 'skill', 'newsletter', 'about-info', 'cta', 'step', 'request-quote', 'features'];
	}

    public function widget_scripts() {

        wp_enqueue_script(
            'logistik-frontend-script',
            LOGISTIK_PLUGDIRURI . 'assets/js/logistik-frontend.js',
            array('jquery'),
            false,
            true
		);

	}


	// public function logistik_regsiter_widget_scripts( ) {

	// 	wp_register_script(
 //            'logistik-tilt',
 //            LOGISTIK_PLUGDIRURI . 'assets/js/tilt.jquery.min.js',
 //            array('jquery'),
 //            false,
 //            true
	// 	);
	// }


    function logistik_elementor_widget_categories( $elements_manager ) {

        $elements_manager->add_category(
            'logistik',
            [
                'title' => __( 'Logistik', 'logistik' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

        $elements_manager->add_category(
            'logistik_footer_elements',
            [
                'title' => __( 'Logistik Footer Elements', 'logistik' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'logistik_header_elements',
            [
                'title' => __( 'Logistik Header Elements', 'logistik' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}
}

Logistik_Extension::instance();