<?php
class Letter_model extends CI_Model
{
    private $genId = null;
    private $image_path;
    private $input_array = [];
    private $division_location_code = [
        '3' => 'KHI',
        '4' => 'ISL',
    ];
    private $_bypass_salary_restriction = false;
    private $page_sizes = array(
        // ISO 216 A Series + 2 SIS 014711 extensions
        'A0' => array(33.11, 46.81),
        'A1' => array(23.39, 33.11),
        'A2' => array(16.54, 23.39),
        'A3' => array(11.69, 16.54),
        'A4' => array(8.27, 11.69),
        'A5' => array(5.83, 8.27),
        'A6' => array(4.13, 5.83),
        'A7' => array(2.91, 4.13),
        'A8' => array(2.05, 2.91),
        'A9' => array(1.46, 2.05),
        'A10' => array(1.02, 1.46),
        'A11' => array(0.71, 1.02),
        'A12' => array(0.51, 0.71),
        // ISO 216 B Series + 2 SIS 014711 extensions
        'B0' => array(39.37, 55.67),
        'B1' => array(27.83, 39.37),
        'B2' => array(19.69, 27.83),
        'B3' => array(13.90, 19.69),
        'B4' => array(9.84, 13.90),
        'B5' => array(6.93, 9.84),
        'B6' => array(4.92, 6.93),
        'B7' => array(3.46, 4.92),
        'B8' => array(2.44, 3.46),
        'B9' => array(1.73, 2.44),
        'B10' => array(1.22, 1.73),
        'B11' => array(0.87, 1.22),
        'B12' => array(0.59, 0.87),
        // ISO 216 C Series + 2 SIS 014711 extensions + 5 EXTENSION
        'C0' => array(36.10, 51.06),
        'C1' => array(25.51, 36.10),
        'C2' => array(18.03, 25.51),
        'C3' => array(12.76, 18.03),
        'C4' => array(9.02, 12.76),
        'C5' => array(6.38, 9.02),
        'C6' => array(4.49, 6.38),
        'C7' => array(3.19, 4.49),
        'C8' => array(2.24, 3.19),
        'C9' => array(1.57, 2.24),
        'C10' => array(1.10, 1.57),
        'C11' => array(0.79, 1.10),
        'C12' => array(0.55, 0.79),
        'C76' => array(3.19, 6.38),
        'DL' => array(4.33, 8.66),
        'DLE' => array(4.49, 8.86),
        'DLX' => array(4.72, 9.25),
        // SIS 014711 E Series
        'E0' => array(34.61, 48.86),
        'E1' => array(24.41, 34.61),
        'E2' => array(17.32, 24.41),
        'E3' => array(12.20, 17.32),
        'E4' => array(8.66, 12.20),
        'E5' => array(6.10, 8.66),
        'E6' => array(4.33, 6.10),
        'E7' => array(3.07, 4.33),
        'E8' => array(2.17, 3.07),
        'E9' => array(1.54, 2.17),
        'E10' => array(1.06, 1.54),
        'E11' => array(0.75, 1.06),
        'E12' => array(0.51, 0.75),
        // SIS 014711 G Series
        'G0' => array(37.72, 53.31),
        'G1' => array(26.65, 37.72),
        'G2' => array(18.86, 26.65),
        'G3' => array(13.31, 18.86),
        'G4' => array(9.41, 13.31),
        'G5' => array(6.65, 9.41),
        'G6' => array(4.69, 6.65),
        'G7' => array(3.31, 4.69),
        'G8' => array(2.32, 3.31),
        'G9' => array(1.65, 2.32),
        'G10' => array(1.14, 1.65),
        'G11' => array(0.83, 1.14),
        'G12' => array(0.55, 0.83),
        // ISO Press
        'RA0' => array(33.86, 48.03),
        'RA1' => array(24.02, 33.86),
        'RA2' => array(16.93, 24.02),
        'RA3' => array(12.01, 16.93),
        'RA4' => array(8.46, 12.01),
        'SRA0' => array(35.43, 50.39),
        'SRA1' => array(25.20, 35.43),
        'SRA2' => array(17.72, 25.20),
        'SRA3' => array(12.60, 17.72),
        'SRA4' => array(8.86, 12.60),
        // German DIN 476
        '4A0' => array(66.22, 93.62),
        '2A0' => array(46.81, 66.22),
        // Variations on the ISO Standard
        'A2_EXTRA' => array(17.52, 24.37),
        'A3+' => array(12.95, 19.02),
        'A3_EXTRA' => array(12.68, 17.52),
        'A3_SUPER' => array(12.01, 20.00),
        'SUPER_A3' => array(12.01, 19.17),
        'A4_EXTRA' => array(9.25, 12.68),
        'A4_SUPER' => array(9.02, 12.68),
        'SUPER_A4' => array(8.94, 14.02),
        'A4_LONG' => array(8.27, 13.70),
        'F4' => array(8.27, 12.99),
        'SO_B5_EXTRA' => array(7.95, 10.87),
        'A5_EXTRA' => array(6.81, 9.25),
        // ANSI Series
        'ANSI_E' => array(34.00, 44.00),
        'ANSI_D' => array(22.00, 34.00),
        'ANSI_C' => array(17.00, 22.00),
        'ANSI_B' => array(11.00, 17.00),
        'ANSI_A' => array(8.50, 11.00),
        // Traditional 'Loose' North American Paper Sizes
        'USLEDGER' => array(17.00, 11.00),
        'LEDGER' => array(17.00, 11.00),
        'ORGANIZERK' => array(11.00, 17.00),
        'BIBLE' => array(11.00, 17.00),
        'USTABLOID' => array(11.00, 17.00),
        'TABLOID' => array(11.00, 17.00),
        'ORGANIZERM' => array(8.50, 11.00),
        'USLETTER' => array(8.50, 11.00),
        'LETTER' => array(8.50, 11.00),
        'USLEGAL' => array(8.50, 14.00),
        'LEGAL' => array(8.50, 14.00),
        'GOVERNMENTLETTER' => array(8.00, 10.50),
        'GLETTER' => array(8.00, 10.50),
        'JUNIORLEGAL' => array(8.00, 5.00),
        'JLEGAL' => array(8.00, 5.00),
        // Other North American Paper Sizes
        'QUADDEMY' => array(35.00, 45.00),
        'SUPER_B' => array(13.00, 19.00),
        'QUARTO' => array(9.00, 11.00),
        'GOVERNMENTLEGAL' => array(8.50, 13.00),
        'FOLIO' => array(8.50, 13.00),
        'MONARCH' => array(7.25, 10.50),
        'EXECUTIVE' => array(7.25, 10.50),
        'ORGANIZERL' => array(5.50, 8.50),
        'STATEMENT' => array(5.50, 8.50),
        'MEMO' => array(5.50, 8.50),
        'FOOLSCAP' => array(8.27, 13.00),
        'COMPACT' => array(4.25, 6.75),
        'ORGANIZERJ' => array(2.75, 5.00),
        // Canadian standard CAN 2-9.60M
        'P1' => array(22.05, 33.86),
        'P2' => array(16.93, 22.05),
        'P3' => array(11.02, 16.93),
        'P4' => array(8.46, 11.02),
        'P5' => array(5.51, 8.46),
        'P6' => array(4.21, 5.51),
        // North American Architectural Sizes
        'ARCH_E' => array(36.00, 48.00),
        'ARCH_E1' => array(30.00, 42.00),
        'ARCH_D' => array(24.00, 36.00),
        'BROADSHEET' => array(18.00, 24.00),
        'ARCH_C' => array(18.00, 24.00),
        'ARCH_B' => array(12.00, 18.00),
        'ARCH_A' => array(9.00, 12.00),
        // -- North American Envelope Sizes
        // - Announcement Envelopes
        'ANNENV_A2' => array(4.37, 5.75),
        'ANNENV_A6' => array(4.75, 6.50),
        'ANNENV_A7' => array(5.25, 7.25),
        'ANNENV_A8' => array(5.50, 8.12),
        'ANNENV_A10' => array(6.25, 9.62),
        'ANNENV_SLIM' => array(3.87, 8.87),
        // - Commercial Envelopes
        'COMMENV_N6_1/4' => array(3.50, 6.00),
        'COMMENV_N6_3/4' => array(3.62, 6.50),
        'COMMENV_N8' => array(3.87, 7.50),
        'COMMENV_N9' => array(3.87, 8.87),
        'COMMENV_N10' => array(4.12, 9.50),
        'COMMENV_N11' => array(4.50, 10.37),
        'COMMENV_N12' => array(4.75, 11.00),
        'COMMENV_N14' => array(5.00, 11.50),
        // - Catalogue Envelopes
        'CATENV_N1' => array(6.00, 9.00),
        'CATENV_N1_3/4' => array(6.50, 9.50),
        'CATENV_N2' => array(6.50, 10.00),
        'CATENV_N3' => array(7.00, 10.00),
        'CATENV_N6' => array(7.50, 10.50),
        'CATENV_N7' => array(8.00, 11.00),
        'CATENV_N8' => array(8.25, 11.25),
        'CATENV_N9_1/2' => array(8.50, 10.50),
        'CATENV_N9_3/4' => array(8.75, 11.25),
        'CATENV_N10_1/2' => array(9.00, 12.00),
        'CATENV_N12_1/2' => array(9.50, 12.50),
        'CATENV_N13_1/2' => array(10.00, 13.00),
        'CATENV_N14_1/4' => array(11.25, 12.25),
        'CATENV_N14_1/2' => array(11.50, 14.50),
        // Japanese (JIS P 0138-61) Standard B-Series
        'JIS_B0' => array(40.55, 57.32),
        'JIS_B1' => array(28.66, 40.55),
        'JIS_B2' => array(20.28, 28.66),
        'JIS_B3' => array(14.33, 20.28),
        'JIS_B4' => array(10.12, 14.33),
        'JIS_B5' => array(7.17, 10.12),
        'JIS_B6' => array(5.04, 7.17),
        'JIS_B7' => array(3.58, 5.04),
        'JIS_B8' => array(2.52, 3.58),
        'JIS_B9' => array(1.77, 2.52),
        'JIS_B10' => array(1.26, 1.77),
        'JIS_B11' => array(0.87, 1.26),
        'JIS_B12' => array(0.63, 0.87),
        // PA Series
        'PA0' => array(33.07, 44.09),
        'PA1' => array(22.05, 33.07),
        'PA2' => array(16.54, 22.05),
        'PA3' => array(11.02, 16.54),
        'PA4' => array(8.27, 11.02),
        'PA5' => array(5.51, 8.27),
        'PA6' => array(4.13, 5.51),
        'PA7' => array(2.76, 4.13),
        'PA8' => array(2.05, 2.76),
        'PA9' => array(1.38, 2.05),
        'PA10' => array(1.02, 1.38),
        // Standard Photographic Print Sizes
        'PASSPORT_PHOTO' => array(1.38, 1.77),
        'E' => array(3.25, 4.72),
        'L' => array(3.50, 5.00),
        '3R' => array(3.50, 5.00),
        'KG' => array(4.02, 5.98),
        '4R' => array(4.02, 5.98),
        '4D' => array(4.72, 5.98),
        '2L' => array(5.00, 7.01),
        '5R' => array(5.00, 7.01),
        '8P' => array(5.98, 7.99),
        '6R' => array(5.98, 7.99),
        '6P' => array(7.99, 10.00),
        '8R' => array(7.99, 10.00),
        '6PW' => array(7.99, 12.01),
        'S8R' => array(7.99, 12.01),
        '4P' => array(10.00, 12.01),
        '10R' => array(10.00, 12.01),
        '4PW' => array(10.00, 15.00),
        'S10R' => array(10.00, 15.00),
        '11R' => array(10.98, 14.02),
        'S11R' => array(10.98, 17.01),
        '12R' => array(12.01, 15.00),
        'S12R' => array(12.01, 17.95),
        // Common Newspaper Sizes
        'NEWSPAPER_BROADSHEET' => array(29.53, 23.62),
        'NEWSPAPER_BERLINER' => array(18.50, 12.40),
        'NEWSPAPER_TABLOID' => array(16.93, 11.02),
        'NEWSPAPER_COMPACT' => array(16.93, 11.02),
        // Business Cards
        'CREDIT_CARD' => array(2.13, 3.37),
        'BUSINESS_CARD' => array(2.13, 3.37),
        'BUSINESS_CARD_ISO7810' => array(2.13, 3.37),
        'BUSINESS_CARD_ISO216' => array(2.05, 2.91),
        'BUSINESS_CARD_IT' => array(2.17, 3.35),
        'BUSINESS_CARD_UK' => array(2.17, 3.35),
        'BUSINESS_CARD_FR' => array(2.17, 3.35),
        'BUSINESS_CARD_DE' => array(2.17, 3.35),
        'BUSINESS_CARD_ES' => array(2.17, 3.35),
        'BUSINESS_CARD_CA' => array(2.01, 3.50),
        'BUSINESS_CARD_US' => array(2.01, 3.50),
        'BUSINESS_CARD_JP' => array(2.17, 3.58),
        'BUSINESS_CARD_HK' => array(2.13, 3.54),
        'BUSINESS_CARD_AU' => array(2.17, 3.54),
        'BUSINESS_CARD_DK' => array(2.17, 3.54),
        'BUSINESS_CARD_SE' => array(2.17, 3.54),
        'BUSINESS_CARD_RU' => array(1.97, 3.54),
        'BUSINESS_CARD_CZ' => array(1.97, 3.54),
        'BUSINESS_CARD_FI' => array(1.97, 3.54),
        'BUSINESS_CARD_HU' => array(1.97, 3.54),
        'BUSINESS_CARD_IL' => array(1.97, 3.54),
        // Billboards
        '4SHEET' => array(40.00, 60.00),
        '6SHEET' => array(47.24, 70.87),
        '12SHEET' => array(120.00, 60.00),
        '16SHEET' => array(80.00, 120.00),
        '32SHEET' => array(160.00, 120.00),
        '48SHEET' => array(240.00, 120.00),
        '64SHEET' => array(320.00, 120.00),
        '96SHEET' => array(480.00, 120.00),
        // -- Old European Sizes
        // - Old Imperial English Sizes
        'EN_EMPEROR' => array(48.00, 72.00),
        'EN_ANTIQUARIAN' => array(31.00, 53.00),
        'EN_GRAND_EAGLE' => array(28.75, 42.00),
        'EN_DOUBLE_ELEPHANT' => array(26.75, 40.00),
        'EN_ATLAS' => array(26.00, 34.00),
        'EN_COLOMBIER' => array(23.50, 34.50),
        'EN_ELEPHANT' => array(23.00, 28.00),
        'EN_DOUBLE_DEMY' => array(22.50, 35.50),
        'EN_IMPERIAL' => array(22.00, 30.00),
        'EN_PRINCESS' => array(21.50, 28.00),
        'EN_CARTRIDGE' => array(21.00, 26.00),
        'EN_DOUBLE_LARGE_POST' => array(21.00, 33.00),
        'EN_ROYAL' => array(20.00, 25.00),
        'EN_SHEET' => array(19.50, 23.50),
        'EN_HALF_POST' => array(19.50, 23.50),
        'EN_SUPER_ROYAL' => array(19.00, 27.00),
        'EN_DOUBLE_POST' => array(19.00, 30.50),
        'EN_MEDIUM' => array(17.50, 23.00),
        'EN_DEMY' => array(17.50, 22.50),
        'EN_LARGE_POST' => array(16.50, 21.00),
        'EN_COPY_DRAUGHT' => array(16.00, 20.00),
        'EN_POST' => array(15.50, 19.25),
        'EN_CROWN' => array(15.00, 20.00),
        'EN_PINCHED_POST' => array(14.75, 18.50),
        'EN_BRIEF' => array(13.50, 16.00),
        'EN_FOOLSCAP' => array(13.50, 17.00),
        'EN_SMALL_FOOLSCAP' => array(13.25, 16.50),
        'EN_POTT' => array(12.50, 15.00),
        // - Old Imperial Belgian Sizes
        'BE_GRAND_AIGLE' => array(27.56, 40.94),
        'BE_COLOMBIER' => array(24.41, 33.46),
        'BE_DOUBLE_CARRE' => array(24.41, 36.22),
        'BE_ELEPHANT' => array(24.25, 30.31),
        'BE_PETIT_AIGLE' => array(23.62, 33.07),
        'BE_GRAND_JESUS' => array(21.65, 28.74),
        'BE_JESUS' => array(21.26, 28.74),
        'BE_RAISIN' => array(19.69, 25.59),
        'BE_GRAND_MEDIAN' => array(18.11, 23.82),
        'BE_DOUBLE_POSTE' => array(17.13, 22.24),
        'BE_COQUILLE' => array(16.93, 22.05),
        'BE_PETIT_MEDIAN' => array(16.34, 20.87),
        'BE_RUCHE' => array(14.17, 18.11),
        'BE_PROPATRIA' => array(13.58, 16.93),
        'BE_LYS' => array(12.48, 15.63),
        'BE_POT' => array(12.09, 15.12),
        'BE_ROSETTE' => array(10.63, 13.66),
        // - Old Imperial French Sizes
        'FR_UNIVERS' => array(39.37, 51.18),
        'FR_DOUBLE_COLOMBIER' => array(35.43, 49.61),
        'FR_GRANDE_MONDE' => array(35.43, 49.61),
        'FR_DOUBLE_SOLEIL' => array(31.50, 47.24),
        'FR_DOUBLE_JESUS' => array(29.92, 44.09),
        'FR_GRAND_AIGLE' => array(29.53, 41.73),
        'FR_PETIT_AIGLE' => array(27.56, 37.01),
        'FR_DOUBLE_RAISIN' => array(25.59, 39.37),
        'FR_JOURNAL' => array(25.59, 37.01),
        'FR_COLOMBIER_AFFICHE' => array(24.80, 35.43),
        'FR_DOUBLE_CAVALIER' => array(24.41, 36.22),
        'FR_CLOCHE' => array(23.62, 31.50),
        'FR_SOLEIL' => array(23.62, 31.50),
        'FR_DOUBLE_CARRE' => array(22.05, 35.43),
        'FR_DOUBLE_COQUILLE' => array(22.05, 34.65),
        'FR_JESUS' => array(22.05, 29.92),
        'FR_RAISIN' => array(19.69, 25.59),
        'FR_CAVALIER' => array(18.11, 24.41),
        'FR_DOUBLE_COURONNE' => array(18.11, 28.35),
        'FR_CARRE' => array(17.72, 22.05),
        'FR_COQUILLE' => array(17.32, 22.05),
        'FR_DOUBLE_TELLIERE' => array(17.32, 26.77),
        'FR_DOUBLE_CLOCHE' => array(15.75, 23.62),
        'FR_DOUBLE_POT' => array(15.75, 24.41),
        'FR_ECU' => array(15.75, 20.47),
        'FR_COURONNE' => array(14.17, 18.11),
        'FR_TELLIERE' => array(13.39, 17.32),
        'FR_POT' => array(12.20, 15.75),
    );
    public function __construct()
    {
        parent::__construct();
        $this->image_path = array_to_path(['application', 'modules', 'incident', 'assets', 'letter_heads']);
    }
    public function get_image_path()
    {
        return $this->image_path;
    }
    public function get_page_sizes()
    {
        return $this->page_sizes;
    }
    public function get_page_size($size)
    {
        return $this->page_sizes[$size];
    }
    public function add($data)
    {
        $questions = null;
        if (!empty($data['question'])) {
            $questions = [];
            foreach ($data['question'] as $q_item) {
                $questions[] = json_decode(base64_decode($q_item));
            }
        }
        $res = $this->db->insert('letters', [
            'title' => $data['title'],
            'body' => $data['body'],
            'footer_note' => trim($data['footer_note']),
            'type' => 'letter',
            'author_type' => $data['author_type'],
            'input_structure' => $questions ? json_encode($questions) : null,
            'requires_admin_approval' => isset($data['requires_admin_approval']) ? 1 : 0,
            'added_by' => $this->session->userdata('id'),
            'updated_by' => $this->session->userdata('id'),
            'added_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        if ($res) {
            $res = $this->db->insert_id();
            $this->activity
                ->set_url('incident/letter/all')
                ->log([
                    'letter_inserted',
                    $data['title'],
                    $res,
                ]);
            return $res;
        }
        return false;
    }
    public function update($id, $data)
    {
        $questions = null;
        if (!empty($data['question'])) {
            $questions = [];
            foreach ($data['question'] as $q_item) {
                $questions[] = json_decode(base64_decode($q_item));
            }
        }
        $res = $this->db->where('id', $id)->update('letters', [
            'title' => $data['title'],
            'footer_note' => trim($data['footer_note']),
            'body' => $data['body'],
            'type' => 'letter',
            'input_structure' => $questions ? json_encode($questions) : null,
            'author_type' => $data['author_type'],
            'requires_admin_approval' => isset($data['requires_admin_approval']) ? 1 : 0,
            'updated_by' => $this->session->userdata('id'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        if ($res) {
            $this->activity
            ->set_url('incident/letter/all')
            ->log([
                'letter_updated',
                $data['title'],
                $id
            ]);
            return $id;
        }
        return false;
    }
    public function delete($id)
    {
        $letter_rec = $this->get_letter($id);
        if($letter_rec){
            $this->activity->log([
                'letter_deleted',
                $letter_rec->title,
                $letter_rec->id,
            ]);
        }
        return $this->db->where(['type' => 'letter', 'id' => $id, 'deleted_at' => null])->update('letters', [
            'deleted_by' => $this->session->userdata('id'),
            'deleted_at' => date('Y-m-d H:i:s'),
        ]);
    }
    public function get_letter($id)
    {
        return $this->db->where(['type' => 'letter', 'id' => $id, 'deleted_at' => null])->get('letters')->row();
    }
    public function get_letter_with_deleted($id)
    {
        return $this->db->where(['type' => 'letter', 'id' => $id])->get('letters')->row();
    }
    public function get_all_letters()
    {
        return $this->db->where(['type' => 'letter', 'deleted_at' => null])->order_by('title', 'asc')->get('letters')->result();
    }
    public function get_all_letters_filtered()
    {
        if ($this->permission->module('generate_manager_letters')->access() || $this->permission->module('generate_hr_letters')->access()) {
            $this->db->group_start();
            if ($this->permission->module('generate_hr_letters')->access()) {
                $this->db->where('author_type', 'hr');
            }
            if ($this->permission->module('generate_manager_letters')->access()) {
                $this->db->or_where('author_type', 'manager');
            }
            $this->db->group_end();
        } elseif ($this->session->userdata('supervisor')) {
            $this->db->where('author_type', 'manager');
        }
        return $this->db->where(['type' => 'letter', 'deleted_at' => null])->order_by('title', 'asc')->get('letters')->result();
    }
    public function generate_pdf($incident_id, $save = false, $path = '')
    {
        $this->load->model('incident_model');

        $incident = $this->incident_model->get($incident_id);
        if (!$incident || !$incident->letter_id) {
            show_404();
        }
        $letter = $this->get_letter_with_deleted($incident->letter_id);
        $this->genId = $incident->genID;
        if ($incident->body) {
            $letter->body = $incident->body;
        } else {
            parse_str($incident->inputs, $this->input_array);
            $letter->body = $this->process_body($letter->body, $incident->employee_name ?? $incident->employee_id, null, $letter->input_structure, $incident_id);
        }

        return $this->toPdf($letter, $incident->employee_name ?? $incident->employee_id, $save, $path, null, $incident);
    }
    public function return_body($incident_id, $bypass_salary_restriction = false)
    {
        $this->_bypass_salary_restriction = $bypass_salary_restriction;
        $this->load->model('incident_model');

        $incident = $this->incident_model->get($incident_id);
        $letter = $this->get_letter_with_deleted($incident->letter_id);
        $this->genId = $incident->genID;
        if ($incident->body) {
            $letter->body = $incident->body;
        } else {
            parse_str($incident->inputs, $this->input_array);
            $letter->body = $this->process_body($letter->body, $incident->employee_name ?? $incident->employee_id, null, $letter->input_structure, $incident_id);
        }

        return $letter->body;
    }
    public function generate_pdf_policy($policy_id, $employee_id, $division_id, $save = false, $path = '')
    {
        $policy = $this->db->where(['type' => 'policy', 'id' => $policy_id, 'deleted_at' => null])->get('letters')->row();
        $policy->body = $this->process_body($policy->body, $employee_id, $division_id, $policy->input_structure, $policy_id);
        return $this->toPdf($policy, null, $save, $path, $division_id);
    }
    public function get_tcpdf_settings()
    {
        $recs = getByWhere('tcpdf_settings');
        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->name] = $rec->value;
        }
        return $data;
    }
    public function get_setting($key)
    {
        return $this->db->where('name', $key)->get('tcpdf_settings')->row();
    }
    public function save_settings($data)
    {
        $this->db->trans_start();
        foreach ($data as $key => $value) {
            $this->db->where('name', $key)->delete('tcpdf_settings');
            $this->db->insert('tcpdf_settings', [
                'name' => $key,
                'value' => $value,
            ]);
        }
        if (count($_FILES)) {
            foreach ($_FILES as $key => $value) {
                if (empty($value['name'])) {
                    continue;
                }
                $chk_exist = $this->db->where('name', $key)->get('tcpdf_settings')->row();
                $file_name = text_to_slug(strtolower(pathinfo($value['name'], PATHINFO_FILENAME))) . '-' . time() . '.' . pathinfo($value['name'], PATHINFO_EXTENSION);
                $moved = move_uploaded_file($value['tmp_name'], FCPATH . array_to_path([$this->image_path, $file_name]));
                if ($moved) {
                    if ($chk_exist) {
                        $this->db->where('name', $key)->update('tcpdf_settings', [
                            'value' => $file_name,
                        ]);
                    } else {
                        $this->db->insert('tcpdf_settings', [
                            'name' => $key,
                            'value' => $file_name,
                        ]);
                    }
                } else {
                    $this->session->set_flashdata('error', sprintf('Unable to upload file "%s", try again', $value['name']));
                    return;
                }
            }
        }
        $this->activity->log([
            'pdf_settings_updated',
            ucfirst($this->uri->segment(4)),
        ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function remove_setting($key)
    {
        $key_data = explode('_', $key);
        if($key_data['1'] == 'image'){
            $this->activity->set_division($key_data[2])->log([
                'pdf_settings_bg_removed',
                $key_data[0] == 'hr' ? 'HR' : 'Manager'
            ]);
            $key_rec = $this->get_setting($key);
            unlink(FCPATH . array_to_path([$this->image_path, $key_rec->value]));
        }
        return $this->db->where('name', $key)->delete('tcpdf_settings');
    }
    public function set_inputs($inputs = null)
    {
        if ($inputs) {
            parse_str($inputs, $this->input_array);
        } elseif ($this->input->post('inputs')) {
            parse_str(base64_decode($this->input->post('inputs')), $this->input_array);
        }
    }
    public function generate_letter_id($employee_id, $division_id = null)
    {
        $id_str = '';
        if (is_numeric($employee_id)) {
            $emp_rec = $this->employee($employee_id);
            $id_str = sprintf('%s/%s/%s/', $this->division_location_code[$emp_rec['division_id']], $emp_rec['hrm_id'], formatted_date_now());
        } else {
            $id_str = sprintf('%s/%s/', $this->division_location_code[$division_id], formatted_date_now());
        }
        $num = 0;
        do {
            $num++;
            $last_rec = $this->db->where('genID', $id_str . $num)->get('incidents')->row();
        } while ($last_rec);
        return $id_str . $num;
    }
    public function add_letter($employee_id, $letter_id, $reason, $inputs)
    {
        $letter_rec = $this->get_letter($letter_id);
        $status = 'pending';
        if ($letter_rec->requires_admin_approval) {
            $status = 'pending-admin';
        }
        $this->db->trans_start();
        $res = $this->db->insert('incidents', [
            'type' => 'letter',
            'letter_id' => $letter_id,
            'employee_id' => is_numeric($employee_id) ? $employee_id : 0,
            'employee_name' => is_numeric($employee_id) ? null : $employee_id,
            'incident_type' => $letter_rec->title,
            'file_path' => null,
            'inputs' => base64_decode($inputs),
            'status' => $status,
            'reason' => $reason,
            'added_by' => $this->session->userdata('id'),
            'added_at' => date('Y-m-d H:i:s'),
            'updated_by' => $this->session->userdata('id'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        if ($res) {
            $res = $this->db->insert_id();

            $this->activity->set_url('incident/details/' . $res);
            if (is_numeric($employee_id)) {
                $this->activity->set_employee($employee_id)
                    ->log([
                        'letter_generated',
                        $letter_rec->title,
                        $res,
                    ]);
            } else {
                $this->activity->log([
                    'letter_generated_outside',
                    $letter_rec->title,
                    $res,
                    $employee_id,
                ]);
            }
        }
        if (is_numeric($employee_id)) {
            $emp_rec = $this->db->where('employee_id', $employee_id)->get('employee_history')->row();
            $this->notify->department('hr')->send(
                sprintf('A new letter "%s" for %s %s (%s) requires your approval', $letter_rec->title, $emp_rec->first_name, $emp_rec->last_name, $emp_rec->hrm_id),
                'incident/pending'
            );
        } else {
            $this->notify->department('hr')->send(
                sprintf('A new letter "%s" for %s requires your approval', $letter_rec->title, $employee_id),
                'incident/pending'
            );
        }
        $this->db->trans_complete();
        return $this->db->trans_status() ? $res : false;
    }
    public function add_incident($employee_id, $subject, $details, $letter_id = null, $inputs = null)
    {
        $status = 'approved';
        if ($letter_id) {
            $letter_rec = $this->get_letter($letter_id);
            $status = 'pending';
            if ($letter_rec->requires_admin_approval) {
                $status = 'pending-admin';
            }
        }
        $this->db->trans_start();
        $res = $this->db->insert('incidents', [
            'type' => 'incident',
            'letter_id' => $letter_id,
            'employee_id' => $employee_id,
            'incident_type' => $subject,
            'description' => $details,
            'file_path' => null,
            'inputs' => base64_decode($inputs),
            'status' => $status,
            'added_by' => $this->session->userdata('id'),
            'added_at' => date('Y-m-d H:i:s'),
            'updated_by' => $this->session->userdata('id'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        if ($res) {
            $res = $this->db->insert_id();

            $this->activity
                ->set_url('incident/details/' . $res)
                ->set_employee($employee_id)
                ->log([
                    'incident_generated',
                    $subject,
                    $res,
                ]);
        }
        $emp_rec = $this->db->where('employee_id', $employee_id)->get('employee_history')->row();
        if ($status == 'approved') {
            $genId = $this->generate_letter_id($employee_id);
            $this->db->where('id', $res)->update('incidents', [
                'genID' => $genId,
            ]);
            $this->notify->department('hr')->send(
                sprintf('A new incident "%s" is added for %s %s (%s)', $subject, $emp_rec->first_name, $emp_rec->last_name, $emp_rec->hrm_id),
                'incident/approved'
            );
            if ($emp_rec->super_visor_id) {
                $this->notify->employee($emp_rec->super_visor_id)->send(
                    sprintf('A new incident "%s" is added for %s %s (%s)', $subject, $emp_rec->first_name, $emp_rec->last_name, $emp_rec->hrm_id),
                    'employee/employees/update_employee_form/' . $employee_id . '#incident_menu'
                );
            }
            $this->notify->employee($employee_id)->send(
                sprintf('A new incident "%s" is added in your profile', $subject),
                'employee/employees/cv/' . $employee_id
            );
        } else {
            $this->notify->department('hr')->send(
                sprintf('A new incident "%s" added for %s %s (%s) needs your approval', $subject, $emp_rec->first_name, $emp_rec->last_name, $emp_rec->hrm_id),
                'incident/pending'
            );
        }
        $this->db->trans_complete();
        return $this->db->trans_status() ? $res : false;
    }
    private function process_body($body, $employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        $data = [];
        $model = $this;
        $body = preg_replace('|(<!--\s*pagebreak\s*-->.*?)\n*(<[\w\d\s]+)|', '$1<span style="page-break-after:always"></span>$2', $body);
        $body = preg_replace_callback(
            "|{{\S+?}}|",
            function ($matches) use ($employee_id, $model, $data, $division_id, $input_structure, $incident_id) {
                $matches_arr = strtolower(trim(trim(strip_tags($matches[0]), '{}')));
                $matches_arr = explode(':', strtolower($matches_arr));

                $matches_arr[1] = preg_replace('#[^\w\d\_]+#is', '_', $matches_arr[1]);
                $matches_arr[1] = strtolower($matches_arr[1]);

                if ($matches_arr && count($matches_arr) == 2) {
                    if (!isset($data[$matches_arr[0]])) {
                        $data[$matches_arr[0]] = call_user_func([$model, $matches_arr[0]], $employee_id, $division_id, $input_structure, $incident_id);
                    }
                    if (isset($data[$matches_arr[0]]) && isset($data[$matches_arr[0]][$matches_arr[1]])) {
                        return $data[$matches_arr[0]][$matches_arr[1]];
                    }
                }
                return $matches[0];
            },
            $body);
        return $body;
    }
    private function toPdf($letter, $employee_id, $save = false, $path = '', $division_id = null, $incident = null)
    {
        $division = $this->division($employee_id, $division_id);
        $employee = $this->employee($employee_id, $division_id);
        $file_name = [];
        if ($employee_id) {
            if (is_numeric($employee_id)) {
                $file_name[] = $employee['hrm_id'];
            } else {
                $file_name[] = $employee_id;
            }
        } elseif ($division_id) {
            $file_name[] = $division['prefix'];
        }
        $file_name[] = time();
        $file_name = implode('-', $file_name) . '.pdf';
        if ($save) {
            $file_name = $path . $file_name;
        }
        $this->load->library('pdfgenerator');
        $this->pdfgenerator->tcpdf($this->load->view('print/letter.php', [
            'letter' => $letter,
            'division' => $division,
        ], true), $file_name, $division['id'], $save, $letter->author_type, $incident);
        return $file_name;
    }
    private function employee($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        if (!is_numeric($employee_id)) {
            return [
                'full_name' => $employee_id,
            ];
        }
        $data = getByWhereAsArray(
            'employee_history,
            position-position.pos_id=employee_history.pos_id-left,
            department-department.dept_id=employee_history.dept_id-left,
            shift-shift.id=employee_history.shift-left,
            gender-gender.id=employee_history.gender-left,
            marital_info-marital_info.id=employee_history.marital_status-left,
            employee_history supervisor-employee_history.super_visor_id=supervisor.employee_id-left,
            employee_history ind_supervisor-employee_history.indirect_super_visor_id=ind_supervisor.employee_id-left',
            'employee_history.*,
            CONCAT_WS(\' \', employee_history.first_name, employee_history.last_name) as full_name,
            CONCAT_WS(\' \', supervisor.first_name, supervisor.last_name) as supervisor,
            CONCAT_WS(\' \', ind_supervisor.first_name, ind_supervisor.last_name) as indirect_supervisor,
            employee_history.ethnic_group as religion,
            position.position_name as designation,
            department.department_name as department,
            gender.gender_name as gender,
            marital_info.marital_sta as marital_status,
            employee_history.ethnic_group as religion,
            shift.name as shift,
            DATE_ADD(employee_history.hire_date, INTERVAL employee_history.probation_period MONTH) as date_of_confirmation',
            ['employee_history.employee_id' => $employee_id],
            [],
            1
        );
        if (!$data) {
            return [];
        }
        $data = decrypt_employee_data($data[0], [], $this->_bypass_salary_restriction);
        if(!empty(trim($data['cnic']))){
            $data['cnic'] = cnic_with_dashes($data['cnic']);
        }
        $col_alter_names = [
            'hire_date' => 'date_of_joining',
            'termination_date' => 'date_of_termination',
            'rate' => 'basic_salary',
            'current_salary' => 'gross_salary',
        ];
        $date_cols = [
            'hire_date',
            'dob',
            'termination_date',
            'date_of_confirmation',
        ];
        foreach ($data as $key => $value) {
            if (in_array($key, $date_cols)) {
                $data[$key] = $value = formatted_date($value);
            }
            if (isset($col_alter_names[$key])) {
                $data[$col_alter_names[$key]] = $value;
            }
        }
        $f = new NumberFormatter("en", NumberFormatter::SPELLOUT);
        if($this->_bypass_salary_restriction || can_decrypt_employee_column('rate')){
            $data['basic_salary_words'] = $f->format($data['basic_salary']);
        } else {
            $data['basic_salary_words'] = '[Encrypted]';
        }
        if($this->_bypass_salary_restriction || can_decrypt_employee_column('current_salary')){
            $data['gross_salary_words'] = $f->format($data['gross_salary']);
        } else {
            $data['gross_salary_words'] = '[Encrypted]';
        }
        if (trim(strtolower($data['gender'])) == 'female') {
            $data['title_prefix'] = 'Ms.';
            $data['relation_of'] = 'D/O';
        } else {
            $data['title_prefix'] = 'Mr.';
            $data['relation_of'] = 'S/O';
        }
        $addr = explode(' ', $data['present_address']);
        $data['address_justified'] = '';
        foreach ($addr as $key => $value) {
            if ($key > 0 && $key % 3 === 0) {
                $data['address_justified'] .= '<br>';
            }
            $data['address_justified'] .= ' ' . $value;
        }
        return $data;
    }
    private function current($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        return [
            'date' => formatted_date_now(),
            'time' => formatted_time_now(),
            'datetime' => formatted_date_now(null, true),
        ];
    }
    private function division($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        $data = null;
        if ($division_id) {
            $data = getByWhereAsArray(
                'divisions',
                '*',
                ['id' => $division_id],
                [],
                1
            );
        } else {
            $data = getByWhereAsArray(
                'employee_history,
                divisions-divisions.id=employee_history.division_id-left',
                'divisions.*',
                ['employee_id' => $employee_id],
                [],
                1
            );
        }
        if (!$data) {
            return [];
        }
        $addr = explode(' ', $data['address']);
        $data['address_justified'] = '';
        foreach ($addr as $key => $value) {
            if ($key > 0 && $key % 3 === 0) {
                $data['address_justified'] .= '<br>';
            }
            $data['address_justified'] .= ' ' . $value;
        }
        return $data[0];
    }
    private function input($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        if ($input_structure) {
            $input_structure = json_decode($input_structure, true);
        }
        $date_cols = [
            'date_of_confirmation',
            'effective_date',
            'period_end',
            'new_period_end',
        ];
        $input_temp = $this->input_array;
        $this->input_array = [];
        foreach ($input_temp as $ikey => $ivalue) {
            $ikey = preg_replace('#[^\w\d\_]+#is', '_', $ikey);
            $ikey = strtolower($ikey);
            $this->input_array[$ikey] = $ivalue;
        }
        foreach ($input_structure as $value_inp) {
            $value_inp['name'] = preg_replace('#[^\w\d\_]+#is', '_', $value_inp['name']);
            $value_inp['name'] = strtolower($value_inp['name']);
            if ($value_inp['type'] == 'multiple_dates') {
                $this->input_array['no_of_days_' . $value_inp['name']] = count(explode(',', $this->input_array[$value_inp['name']]));
            }
            if ($value_inp['type'] == 'textarea') {
                $this->input_array[$value_inp['name']] = $this->remove_first_p($this->input_array[$value_inp['name']]);
            }
        }
        if (isset($this->input_array['no_of_days'])) {
            $this->input_array['days_range_end'] = date('d-m-Y', strtotime('+' . $this->input_array['no_of_days'] . ' day'));
        }
        if (isset($this->input_array['period_end']) && isset($this->input_array['new_period_end'])) {
            $olddate = new DateTime($this->input_array['period_end']);
            $newdate = new DateTime($this->input_array['new_period_end']);
            $diffdate = $olddate->diff($newdate);
            $this->input_array['interval_months'] = $diffdate->format('%m');
            $this->input_array['interval_months'] .= $this->input_array['interval_months'] < 2 ? ' month' : ' months';
        }
        if (isset($this->input_array['reason']) && $this->input_array['reason'] == 'others') {
            $this->input_array['reason'] = $this->input_array['others'];
        }
        foreach ($this->input_array as $key => $value) {
            if (in_array($key, $date_cols) || preg_match('|^\d{2,4}\-\d{2}\-\d{2}$|', $value)) {
                $this->input_array[$key] = formatted_date($value);
            }
        }
        if (isset($this->input_array['bank_name'])) {
            $this->input_array['bank_location'] = $this->input_array['bank_name'] == 'Allied Bank Ltd' ? 'Islamabad/Rawalpindi' : 'Karachi';
        }
        return $this->input_array;
    }
    private function letter($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        if (empty($incident_id)) {
            return [
                'id' => $this->genId,
            ];
        }
        $letter_data = $this->db->select(
            'incidents.incident_type,
            COALESCE(incidents.description, incidents.reason) as reason,
            letters.title as title')
            ->where('incidents.id', $incident_id)
            ->join('letters', 'letters.id=incidents.letter_id', 'left')
            ->get('incidents')->row_array();
        $letter_data['reason'] = $this->remove_first_p($letter_data['reason']);
        return array_merge([
            'id' => $this->genId,
        ], $letter_data);
    }
    private function policy($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        if (empty($incident_id)) {
            return [];
        }
        return $this->db->select(
            'title as title')
            ->where('id', $incident_id)
            ->get('letters')->row_array();
    }
    private function author($employee_id, $division_id = null, $input_structure = null, $incident_id = null)
    {
        if ($this->session->userdata('id') == 1 && $this->session->userdata('isAdmin')) {
            return [
                'department' => 'Human Resources',
            ];
        } elseif (!$this->session->userdata('employee_id')) {
            return [];
        }
        return $this->employee($this->session->userdata('employee_id'));
    }
    public function view($incident_id)
    {
        $this->generate_pdf($incident_id);
    }
    public function test($id)
    {
        $emp = $this->db->limit(1)->get('employee_history')->row();
        $letter = $this->get_letter($id);
        $this->genId = $this->generate_letter_id($emp->employee_id);
        $letter->body = $this->process_body($letter->body, $emp->employee_id);

        return $this->toPdf($letter, $emp->employee_id);
    }
    public function get_table_names()
    {
        $allowed_tables = [
            'city',
            'country',
            'department',
            'divisions',
            'gender',
            'payroll_holiday',
            'leave_type',
            'position',
        ];
        $data = array_map(function ($val) use ($allowed_tables) {
            if (in_array($val, $allowed_tables)) {
                return ['key' => $val, 'value' => slug_to_readable($val)];
            }
            return null;
        }, $this->db->list_tables());
        return array_filter($data);
    }
    public function get_table_fields($table_name)
    {
        $allowed_columns = [
            'city' => [
                'Name',
                'District',
            ],
            'country' => [
                'Code',
                'Name',
                'Continent',
                'Region',
            ],
            'department' => [
                'department_name',
            ],
            'divisions' => [
                'name',
                'email',
                'phone_no',
                'address',
            ],
            'gender' => [
                'gender_name',
            ],
            'payroll_holiday' => [
                'holiday_name',
            ],
            'leave_type' => [
                'leave_type',
            ],
            'position' => [
                'position_name',
            ],
        ];
        $data = array_values(array_filter($this->db->list_fields($table_name), function ($val) use ($allowed_columns, $table_name) {
            return in_array($val, $allowed_columns[$table_name]);
        }));
        return array_map(function ($val) {
            return ['value' => $val, 'text' => slug_to_readable($val)];
        }, $data);
    }
    public function get_tag_suggestions()
    {
        $employee_fields = [
            'hrm_id',
            'prefix',
            'first_name',
            'middle_name',
            'last_name',
            'email',
            'phone',
            'alter_phone',
            'present_address',
            'parmanent_address',
            'father_name',
            'degree_name',
            'institute_name',
            'cgp',
            'passing_year',
            'state',
            'city',
            'dob',
            'home_email',
            'business_email',
            'home_phone',
            'business_phone',
            'cell_phone',
            'emerg_contct',
            'probation_period',
            'performance_duration',
            'resignation_date',
            'cnic',
            'eobi',
            'work_from_home',
            'transportation_required',
            'primary_project',
            'team',
            'campaign',
            'pseudo',
            'medflow_id',
            'skype_id',
            'replacement',
            'last_employer',
            'last_employer_address',
            'account_title',
            'account_number',
            'branch_code',
            'branch_address',
            'full_name',
            'supervisor',
            'indirect_supervisor',
            'designation',
            'department',
            'marital_status',
            'religion',
            'gender',
            'date_of_confirmation',
            'date_of_joining',
            'date_of_termination',
            'basic_salary',
            'basic_salary_words',
            'gross_salary',
            'gross_salary_words',
            'title_prefix',
            'relation_of',
            'address_justified',
        ];

        sort($employee_fields);

        $suggestions = array_map(function ($val) {
            return 'employee:' . $val;
        }, $employee_fields);

        $suggestions = array_merge($suggestions, [
            /* Letter */
            'letter:id',
            'letter:title',
            'letter:reason',
            'letter:incident_type',
            /* Current */
            'current:date',
            'current:datetime',
            'current:time',
            /* Division */
            'division:name',
            'division:prefix',
            'division:email',
            'division:phone_no',
            'division:address',
            'division:address_justified',
        ]);

        $suggestions = array_merge($suggestions, array_map(function ($val) {
            return 'author:' . $val;
        }, $employee_fields));
        return $suggestions;
    }
    private function remove_first_p($html)
    {
        return preg_replace('~^<p.*?>(.*?)</p>~is', '$1', $html, 1);
    }
}
