<?php
use PHPMailer\PHPMailer\PHPMailer;

class TemplateMailer
{
    private $CI;
    private $mail;
    private $config_arr = [];
    private $logpath = APPPATH . 'logs/mail/';
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->mail = new PHPMailer();
        $this->mail->SMTPDebug = false;
        $this->mail->XMailer = ' ';
        if (!is_dir($this->logpath)) {
            mkdir($this->logpath);
        }
    }
    public function config($config = 0)
    {
        $this->config_arr = $this->CI->db->select('email_templates.*, divisions.name as division_name')->where('division_id', $config)->join('divisions', 'email_templates.division_id=divisions.id', 'left')->get('email_templates')->row_array();
        if ($this->config_arr['isSMTP']) {
            $this->mail->isSMTP(); //Send using SMTP
            $this->mail->SMTPAuth = true; //Enable SMTP authentication
            $this->mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS; //Enable TLS encryption; `PHPMailer::ENCRYPTION_SMTPS` encouraged
        }
        $this->mail->Host = $this->config_arr['host']; //Set the SMTP server to send through
        $this->mail->Username = $this->config_arr['username']; //SMTP username
        $this->mail->Password = $this->CI->encryption->decrypt($this->config_arr['password']); //SMTP password
        $this->mail->Port = $this->config_arr['port']; //TCP port to connect to, use 465 for `PHPMailer::ENCRYPTION_SMTPS` above
        $this->mail->isHTML(true);
        return $this;
    }
    public function get_config($key)
    {
        return $this->config_arr[$key];
    }

    public function send_to_hr($division, $subject, $body, $cc = [], $bcc = [], $all_division_hr_staff = true, $all_hr_staff = false)
    {
        $this->config($division);
        if (empty(trim($this->config_arr['email_hr']))) {
            $this->log(sprintf('No HR email configured. Failed to send email to hr with subject "%s"', $subject));
            return false;
        }
        if ($all_division_hr_staff) {
            if (!is_array($cc)) {
                $cc = [$cc];
            }
            
            $this->CI->load->model('Employee/Employees_model');

            $hr_emails = $this->CI->Employees_model
                ->get_employee_by_user_id($this->CI->notify->department('hr', !$all_hr_staff ? $division : null)->get_all_users(), 'employee.business_email');
            $this->CI->notify->clear();
            if ($hr_emails) {
                $hr_emails = array_column($hr_emails, 'business_email');
                $cc = array_merge($cc, $hr_emails);
            }

            $cc = array_unique($cc);
        }
        return $this->_send($this->config_arr['email_hr'], null, $subject, $body, $cc, $bcc);
    }
    public function send_to_department($division, $department, $subject, $body, $cc = [], $bcc = [], $all_staff = false)
    {
        $to = [];
        $this->config($division);
        if (empty(trim($this->config_arr['email_hr']))) {
            $this->log(sprintf('No HR email configured. Failed to send email to hr with subject "%s"', $subject));
            return false;
        }

        $this->CI->load->model('Employee/Employees_model');
        $all_users = $this->CI->notify->department($department, !$all_staff ? $division : null)->get_all_users();
        if(!$all_users) {
            return false;
        }
        $hr_emails = $this->CI->Employees_model
            ->get_employee_by_user_id($all_users, 'employee.business_email');
        $this->CI->notify->clear();
        if ($hr_emails) {
            $hr_emails = array_column($hr_emails, 'business_email');
            $to = array_merge($to, $hr_emails);
        }
        $to = array_unique($to);

        return $this->_send($to, null, $subject, $body, $cc, $bcc);
    }

    public function send($job_id, $template_category, $to, $cc = [], $bcc = [], $candidate_id = null, $employee_id = null, $additional_can_param = [])
    {
        $job = $this->CI->db->where('jp_id', $job_id)->get('job_posting')->row();
        $division_id = $job->division_id;
        $this->config($division_id);
        $template = $this->get_template($division_id, $template_category);
        if (!$template) {
            $this->log("Template '$template_category' not found for division id $division_id");
            return false;
        }
        $this->template_category = $template_category;
        $candidate = $this->get_candidate($candidate_id);
        $employee = $employee_id;

        $body = $template->template;
        $body = preg_replace_callback(
            "|{{\S+?}}|",
            function ($matches) use ($candidate, $employee, $additional_can_param) {
                $matches = strtolower(trim(trim(strip_tags($matches[0]), '{}')));
                $matches = explode(':', strtolower($matches));
                if (isset($additional_can_param[$matches[0]]) && isset($additional_can_param[$matches[0]][$matches[1]])) {
                    return $additional_can_param[$matches[0]][$matches[1]];
                } elseif ($matches[0] == 'candidate' && isset($candidate[$matches[1]])) {
                    return $candidate[$matches[1]];
                } elseif ($matches[0] == 'employee' && isset($employee->{$matches[1]})) {
                    return $employee->{$matches[1]};
                }
            },
            $body);
        $template->subjects = preg_replace_callback(
            "|{{\S+?}}|",
            function ($matches) use ($candidate, $employee, $additional_can_param) {
                $matches = strtolower(trim(trim(strip_tags($matches[0]), '{}')));
                $matches = explode(':', strtolower($matches));
                if (isset($additional_can_param[$matches[0]]) && isset($additional_can_param[$matches[0]][$matches[1]])) {
                    return $additional_can_param[$matches[0]][$matches[1]];
                } elseif ($matches[0] == 'candidate' && isset($candidate[$matches[1]])) {
                    return $candidate[$matches[1]];
                } elseif ($matches[0] == 'employee' && isset($employee->{$matches[1]})) {
                    return $employee->{$matches[1]};
                }
            },
            $template->subjects);

        if ($template->cc_hr) {
            $cc[] = $this->config_arr['email_hr'];
            $cc = array_unique($cc);
        }
        return $this->_send($to ? $to : $candidate['email'], null, $template->subjects, $body, $cc, $bcc);
    }

    public function test_email_settings($to, $subject, $body)
    {
        return $this->__send($to, $this->config_arr['email_from'], $subject, $body, $this->config_arr['email_hr']);
    }

    public function _send($to, $from = null, $subject, $body, $cc = [], $bcc = [], $attachments = [])
    {
        if (!is_array($to)) {
            $to = [$to];
        }
        if (!is_array($cc)) {
            $cc = [$cc];
        }
        if (!is_array($bcc)) {
            $bcc = [$bcc];
        }
        if (!is_array($attachments)) {
            $attachments = [$attachments];
        }
        return $this->CI->db->insert('email_queue', [
            'division_id' => $this->config_arr['division_id'],
            'eto' => implode(',', $to),
            'efrom' => $from ?? $this->config_arr['email_from'],
            'subject' => $subject,
            'content' => json_encode(['cc' => implode(',', $cc), 'bcc' => implode(',', $bcc), 'body' => $body, 'attachments' => json_encode($attachments)]),
            'added_at' => date('Y-m-d H:i:s'),
        ]);
    }

    public function __send($to, $from, $subject, $body, $cc = [], $bcc = [], $attachments = [])
    {
        if ($this->template_category) {
            $this->log(sprintf('Sending email for event %s', $this->template_category));
        } else {
            $this->log('Sending email');
        }

        $this->mail->setFrom($from, $this->config_arr['division_name']);
        if (is_array($cc)) {
            foreach ($cc as $ccc) {
                $this->mail->addCC($ccc);
            }
        } else {
            $this->mail->addCC($cc);
        }

        if (is_array($bcc)) {
            foreach ($bcc as $bccc) {
                $this->mail->addBCC($bccc);
            }
        } else {
            $this->mail->addBCC($bcc);
        }

        if (is_array($to)) {
            foreach ($to as $tat) {
                $this->mail->addAddress($tat);
            }
        } else {
            $this->mail->addAddress($to);
        }

        foreach ($attachments as $key => $attachment) {
            $this->mail->AddAttachment($attachment, is_numeric($key) ? '' : $key);
        }

        $this->mail->Subject = $subject;
        $this->mail->Body = $body;
        try {
            $res = $this->mail->send();
            if ($res) {
                $this->log('Email sent successfully');
                return true;
            } else {
                throw new Exception($this->mail->ErrorInfo);
            }
        } catch (\Throwable $e) {
            $this->log($e->getMessage());
            $this->CI->session->set_userdata('email_exception', $e->getMessage());
            return false;
        } finally {
            $this->mail = new PHPMailer();
        }
    }
    private function log($message)
    {
        @file_put_contents(
            $this->logpath . date('Ymd') . '.log',
            sprintf("%s| %s\n", date('c'), $message),
            FILE_APPEND
        );
    }
    private function get_template($division_id, $temp_cat)
    {
        return $this->CI->db->select('document_template.*, template_category.cc_hr')
            ->join('template_category', 'template_category.id=document_template.category_id', 'left')
            ->where('template_category.keyword', $temp_cat)
            ->where('document_template.division_id', $division_id)
            ->get('document_template')->row();
    }
    private function get_candidate($id)
    {
        $rec = $this->CI->db->select(
            'candidate_basic_info.first_name,
            candidate_basic_info.last_name,
            candidate_basic_info.email,
            candidate_basic_info.phone,
            candidate_basic_info.alter_phone as alternative_phone,
            candidate_basic_info.address,
            candidate_basic_info.total_years_of_experience as experience,
            candidate_basic_info.applied_on,
            position.position_name as position'
        )
            ->join('job_posting', 'job_posting.jp_id=candidate_basic_info.jp_id', 'left')
            ->join('position', 'position.pos_id=job_posting.pos_id', 'left')
            ->where('candidate_basic_info.can_id', $id)
            ->get('candidate_basic_info')->row_array();

        $rec['applied_on'] = formatted_date($rec['applied_on']);

        $rec['interview_date'] = $this->CI->db->select('candidate_shortlist.interview_date')
            ->where('candidate_shortlist.can_id', $id)
            ->where('candidate_shortlist.interview_done', 0)
            ->get('candidate_shortlist')->row_array()['interview_date'];
        $rec['interview_date'] = formatted_date($rec['interview_date'], true);
        return $rec;
    }
}
