<?php

namespace App\Jobs;

use App\Mail\BulkPdfMail;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use App\Helpers\GeneralHelper;

use ZipArchive;

class ProcessBulkPdf implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;
    protected $is_cascade;
    protected $bulk_report;
    protected $userEmail;


    public $tries = 5;
    public $timeout = 1200;

    /**
     * Create a new job instance.
     *
     * @return void
     */

    public function __construct($is_cascade, $bulk_report, $userEmail)
    {
        $this->is_cascade = $is_cascade;
        $this->bulk_report = $bulk_report;
        $this->userEmail = $userEmail;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $zip = new ZipArchive;
        $zipFileName = 'bulk_reports_' . time() . '.zip';
        $zipFilePath = Storage::path($zipFileName);
        $pdfFilePaths = [];

        if ($zip->open($zipFilePath, ZipArchive::CREATE) === TRUE) {

            if ($this->is_cascade) {
                $mappedReport = GeneralHelper::mapCascadeReportKeys($this->bulk_report);
                foreach ($mappedReport as $index => $report) {


                    // Generate PDF File
                    $pdf = Pdf::loadView('services.physician.single-report-pdf', ['single_reports' => $report]);
                    $patient = $report[0]['patient_id'];
                    $pdfFileName = "patient_".$patient."_"."report".$index.'.pdf';
                    $pdfFilePath = Storage::path($pdfFileName);
                    Storage::put($pdfFileName, $pdf->output());
                    $zip->addFile($pdfFilePath, $pdfFileName);
                    $pdfFilePaths[] = $pdfFilePath;
                }
            } else {
                foreach ($this->bulk_report as $index => $report) {

                    $mappedReport = GeneralHelper::mapBulkReportKeys($report);

                    // Generate PDF File
                    $pdf = Pdf::loadView('services.physician.single-report-pdf', ['single_reports' => array($mappedReport)]);
                    $cptcode = strtolower(explode(':', $mappedReport['Procedure Code'])[0]);
                    $pdfFileName = "cpt_".$cptcode."_"."report".$index.'.pdf';
                    $pdfFilePath = Storage::path($pdfFileName);
                    Storage::put($pdfFileName, $pdf->output());
                    $zip->addFile($pdfFilePath, $pdfFileName);
                    $pdfFilePaths[] = $pdfFilePath;

                }
            }

            $zip->close();
        }


        Mail::to($this->userEmail)->send(new BulkPdfMail($zipFilePath));

        //delete zip file
        unlink($zipFilePath);

        // Delete the PDF files
        foreach ($pdfFilePaths as $pdfFilePath) {
            unlink($pdfFilePath);
        }
    }
}
