<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Api\BaseController;
use App\Http\Controllers\HomeController;
use Illuminate\Http\Request;
use Validator;

class OmfsApiController extends BaseController
{

    private function validateSingleReport(Request $request)
    {
        // Validation rules for both required and conditional fields
        $rules = [
            'service_name' => 'required|in:Physician Services',
            'dos' => 'required|date',
            'is_cascade' => 'required|boolean',
            'servicetype' => 'required|in:Facility,Non-Facility',
            'providertype' => 'required|in:Physician,Non-Physician,Clinical Soial Worker',
            'conversionfactor' => 'nullable|string',
            'locality_type' => 'required|in:Zip-Code,County',
            'locality_state' => 'required',
            'localityzip' => 'required_if:locality_type,Zip-Code|nullable|numeric',
            'locality_number' => 'required_if:locality_type,County|nullable|numeric',
            'locality_name' => 'required_if:locality_type,County|nullable|string',
            'singleReport' => 'required|array|min:1',
            'singleReport.*.cptcode' => 'required|string',
            'singleReport.*.modifiers' => 'nullable|string',
            'singleReport.*.units' => 'required|integer|min:1',
            'singleReport.*.payment_total' => 'nullable|numeric',
        ];

        // Validate the request
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $errors = [];
            foreach ($validator->errors()->toArray() as $key => $messages) {
                $errors[$key] = $messages[0];
            }

            return [
                'success' => false,
                'errors' => $errors,
            ];
        }

        return ['success' => true];
    }

    public function singleReportLoadQuery(Request $request)
    {

        // Perform validation
        $validationResult = $this->validateSingleReport($request);

        if (!$validationResult['success']) {
            return response()->json($validationResult,422);
        }


        $single_report = HomeController::single_report_load_query($request, true);

        // Check if the response is a JsonResponse object
        if ($single_report instanceof \Illuminate\Http\JsonResponse) {
            $single_report = $single_report->getData(true); // Get the data as an associative array
        }

        // Check for errors in the data
        if (isset($single_report['error'])) {
            return $this->sendError($single_report['error']);

        }

        // Handle cases where the response is empty
        if (count($single_report) == 0) {
            return $this->sendError("Something Went Wrong!");
        }
        $feeSchedules = [];
        foreach ($single_report as $key => $report) {
            if (isset($report['Fee Schedule'])) {
                if (isset($report['r_hcpcs']) && isset($report['Fee Schedule'])) {

                    if (is_numeric($report['Fee Schedule'])) {
                        $feeSchedules[$key][$report['r_hcpcs']] = number_format($report['Fee Schedule'], 2);
                    } else {
                        $feeSchedules[$key][$report['r_hcpcs']] = $report['Fee Schedule']; // Or handle non-numeric values as needed
                    }

                } else {
                    if (is_numeric($report['Fee Schedule'])) {
                        $feeSchedules[$key][$report['Procedure Code']] = number_format($report['Fee Schedule'], 2);
                    } else {
                        $feeSchedules[$key][$report['Procedure Code']] = $report['Fee Schedule']; // Or handle non-numeric values as needed
                    }
                }

            }
        }

        // Handle cases where no Fee Schedule is found
        if (empty($feeSchedules)) {
            return $this->sendError("No Fee Schedule found in the report.");
        }

        // If single Fee Schedule, return it directly; otherwise, return all
        if (count($feeSchedules) === 1) {
            return $this->sendResponse("Single Cpt Value Fetched", $feeSchedules);
        }

        return $this->sendResponse("Multiple Cpt Values Fetched", $feeSchedules);
    }


    public function GetSingleReportPdf(Request $request)
    {
        // Perform validation
        $validationResult = $this->validateSingleReport($request);

        if (!$validationResult['success']) {
            return response()->json($validationResult,422);
        }

        $single_report = HomeController::single_report_pdf($request, true);


        // Handle cases where no Fee Schedule is found
        if (empty($single_report)) {
            return $this->sendError("Failed to Generate Pdf.");
        }


        return $single_report;
    }



}
