<?php

use Google\Service\Forms\FormSettings;
use Google\Service\Forms\QuizSettings;

class Quiz_model extends CI_Model
{
    private $logpath = APPPATH . 'logs/gforms/';
    public $googleClient = null;
    public $mailClient = null;
    public function __construct()
    {
        parent::__construct();

        $this->googleClient = new GoogleClient('https:' . preg_replace('#^[https|http]+\:#i', '', base_url('quiz/login')));
        $this->mailClient = new GoogleClient('https:' . preg_replace('#^[https|http]+\:#i', '', base_url('quiz/mail_login')), 'form_email');
    }
    public function getActiveQuiz()
    {
        $recs = $this->db->where('due_date >', date('Y-m-d H:i:s', strtotime('-2 months')))->get('google_quiz_detail')->result();
        return $recs;
    }
    public function getOpenQuizEmployee($employee_id = null)
    {
        if (!$employee_id) {
            $employee_id = $this->session->userdata('employee_id');
        }
        if (!$employee_id) {
            return [];
        }
        $recs = $this->db
            ->select('google_quiz_detail.*, qz_rel_form_employee.resultSent, qz_rel_form_employee.totalScore as obtained, qz_rel_form_employee.submittedAt as takenAt')
            ->join('qz_rel_form_employee', 'google_quiz_detail.id=qz_rel_form_employee.form_id', 'left')
            ->where([
                'google_quiz_detail.due_date >' => date('Y-m-d H:i:s'),
                'qz_rel_form_employee.employee_id' => $employee_id,
                'qz_rel_form_employee.deletedAt' => null,
            ])
            ->order_by('due_date', 'asc')->get('google_quiz_detail')->result();
        return $recs;
    }
    public function obtainLink($form_id)
    {
        $employee_id = $this->session->userdata('employee_id');
        $rec = $this->db
            ->select('qz_form.link')
            ->join('qz_rel_form_employee', 'qz_form.id=qz_rel_form_employee.form_id', 'left')
            ->where([
                'qz_form.due_date >' => date('Y-m-d H:i:s'),
                'qz_form.id' => $form_id,
                'qz_rel_form_employee.employee_id' => $employee_id,
                'qz_rel_form_employee.deletedAt' => null,
            ])->get('qz_form')->row();
        return $rec ? $rec->link : null;
    }
    public function addQuiz($post)
    {
        $form = $this->getFormDetails($post['quiz_id']);
        $this->db->trans_start();
        $this->db->insert('qz_form', $quiz_data = [
            'form_id' => $post['quiz_id'],
            'label' => $post['label'],
            'autorelease-score' => intval($post['autorelease-score']),
            'record_purposes' => $post['record_purposes'] == 1 ? 1 : 0,
            'title' => $form->info->documentTitle,
            'description' => $form->info->description,
            'link' => $form->responderUri,
            'due_date' => sql_date($post['due_date']) . ' 23:59:59',
            'passing_perc' => $post['passing_perc'],
            'added_at' => date('Y-m-d H:i:s'),
        ]);
        $email_time = $post['send'] == 'now' ? date('Y-m-d H:i:s') : date('Y-m-d H:i:s', strtotime($post['email_scheduled']));
        $form_id = $this->db->insert_id();
        $this->activity->set_url('quiz/details/' . $form_id)->log([
            'quiz_created',
            $quiz_data['label'],
            $quiz_data['title'],
            $form_id,
        ]);
        $email_content = [
            'body' => $post['body'],
            'bcc' => $post['bcc'],
        ];
        foreach ($post['emps'] as $emp) {
            $emp_rec = $this->db->select('business_email, direct_manager_email')->where('employee_id', $emp)->get('employee_details')->row();
            if (!$emp_rec) {
                continue;
            }
            $this->db->insert('qz_rel_form_employee', [
                'form_id' => $form_id,
                'employee_id' => $emp,
                'email' => $emp_rec->business_email,
            ]);
            if ($post['record_purposes'] != 1) {
                $this->db->insert('qz_email_queue', [
                    'form_id' => $form_id,
                    'employee_id' => $emp,
                    'eto' => $emp_rec->business_email,
                    'subject' => $post['subject'],
                    'content' => json_encode(array_merge($email_content, ['cc' => [$emp_rec->direct_manager_email]])),
                    'send_at' => $email_time,
                    'added_at' => date('Y-m-d H:i:s'),
                ]);
            }
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function editQuiz($id, $post)
    {
        $this->db->trans_start();
        $this->db->where('id', $id)->update('qz_form', [
            'label' => $post['title'],
            'autorelease-score' => intval($post['autorelease-score']),
            'due_date' => sql_date($post['due_date']) . ' 23:59:59',
            'passing_perc' => $post['passing_perc'],
        ]);
        $this->activity->set_url('quiz/details/' . $id)->log([
            'quiz_updated',
            $post['title'],
            $id,
        ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function getQuizList()
    {
        if (!$quizList = $this->cache->get('system.quiz.list')) {
            $drive = $this->googleClient->getDriveInstance();
            $file_params = [
                'id',
                'name',
                'mimeType',
                'webViewLink',
            ];

            $nextPageToken = null;
            $quizList = [];
            do {
                $optParams = array(
                    'q' => "mimeType='application/vnd.google-apps.form' and trashed = false",
                    'pageSize' => 20,
                    'fields' => 'nextPageToken, files(' . implode(', ', $file_params) . ')',
                    'pageToken' => $nextPageToken,
                );
                $results = $drive->files->listFiles($optParams);
                foreach ($results->getFiles() as $file) {
                    $form = $this->getFormDetails($file->id);
                    $quizList[] = [
                        'id' => $file->id,
                        'label' => $form->info->documentTitle,
                        'isQuiz' => (($form->settings instanceof FormSettings) && ($form->settings->quizSettings instanceof QuizSettings) && $form->settings->quizSettings->isQuiz),
                    ];
                }
                $nextPageToken = $results->nextPageToken;
            } while (!is_null($nextPageToken));
            $this->cache->save('system.quiz.list', $quizList, 30);
        }

        // $quizList = array_filter($quizList, function ($q) {
        //     return $this->db->where(['form_id' => $q['id'], 'due_date >=' => date('Y-m-d H:i:s')])->count_all_results('qz_form') < 1;
        // });
        return $quizList;
    }
    public function getQuizDetails($id)
    {
        $quiz = $this->db->where('id', $id)->get('google_quiz_detail')->row();
        if ($quiz) {
            $quiz->questions = $this->db->where('form_id', $id)->get('qz_question')->result();
        }
        return $quiz;
    }
    public function getFormDetails($form_id)
    {
        $formInstance = $this->googleClient->getFormInstance();

        return $formInstance->forms->get($form_id);
    }
    public function getEmployees()
    {
        $departments = $this->options->get_system('system.quiz.departments');
        $q = $this->db->select('employee_id, hrm_id, pseudo, full_name, position, hire_date, business_email, direct_manager_name, tenure, tenure_in_months');
        if ($departments) {
            $q->where_in('dept_id', explode(',', $departments));
        } elseif ($this->session->userdata('isAdmin')) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model');
            $subordinates = $this->employees_model->get_subordinates();
            if ($subordinates) {
                $subordinates = array_column($subordinates, 'employee_id');
                $q->where_in('employee_id', $subordinates);
            } else {
                $q->where('employee_id', 'z');
            }
        } else {
            $q->where('employee_id', 'z');
        }
        $q->where('is_super_visor', 0);
        $q->where_in('status', ['active', 'on-hold', 'resignation-received']);
        return $q->get('employee_details')->result();
    }
    public function getManagers()
    {
        $departments = $this->options->get_system('system.quiz.departments');
        $q = $this->db->select('employee_id, hrm_id, pseudo, full_name, position, hire_date, business_email, direct_manager_name, tenure, tenure_in_months');
        if ($departments) {
            $q->where_in('dept_id', explode(',', $departments));
        } elseif ($this->session->userdata('isAdmin')) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model');
            $subordinates = $this->employees_model->get_subordinates();
            if ($subordinates) {
                $subordinates = array_column($subordinates, 'employee_id');
                $q->where_in('employee_id', $subordinates);
            } else {
                $q->where('employee_id', 'z');
            }
        } else {
            $q->where('employee_id', 'z');
        }
        $q->where('is_super_visor', 1);
        $q->where_in('status', ['active', 'on-hold', 'resignation-received']);
        return $q->get('employee_details')->result();
    }
    public function getEmployeesById($emp_ids)
    {
        $emp_ids = array_filter($emp_ids);
        if (!$emp_ids) {
            return null;
        }
        return $this->db->select('business_email, direct_manager_email')->where_in('employee_id', $emp_ids)->get('employee_details')->result();
    }
    public function syncForms()
    {
        $drive = $this->googleClient->getDriveInstance();
        $file_params = [
            'id',
            'name',
            'mimeType',
            'webViewLink',
            'modifiedTime',
            'version',
        ];
        $nextPageToken = null;
        $quizList = [];
        $lastUpdate = $this->db->select_max('modifiedTime')->where('due_date >', date('Y-m-d H:i:s'))->get('qz_form')->row();
        if ($lastUpdate && $lastUpdate->modifiedTime) {
            $lastUpdate = date('c', strtotime($lastUpdate->modifiedTime));
            do {
                $optParams = array(
                    'q' => "mimeType='application/vnd.google-apps.form' and trashed = false and modifiedTime > '{$lastUpdate}'",
                    'pageSize' => 20,
                    'fields' => 'nextPageToken, files(' . implode(', ', $file_params) . ')',
                    'pageToken' => $nextPageToken,
                );
                $results = $drive->files->listFiles($optParams);
                foreach ($results->getFiles() as $file) {
                    $form = $this->getFormDetails($file->id);
                    if (!(($form->settings instanceof FormSettings) && ($form->settings->quizSettings instanceof QuizSettings) && $form->settings->quizSettings->isQuiz)) {
                        continue;
                    }
                    $quizList[] = $form;
                }
                $nextPageToken = $results->nextPageToken;
            } while (!is_null($nextPageToken));
        }
        $this->db->trans_start();
        if ($quizList) {
            $quizList = array_filter($quizList, function ($qz) {
                return $this->db->where(['form_id' => $qz->formId])->count_all_results('qz_form') > 0;
            });
            foreach ($quizList as $quiz) {
                $form = $this->db->where('form_id', $quiz->formId)->get('qz_form')->row();
                $this->db->where('form_id', $form->id)->delete('qz_question');
                foreach ($quiz->items as $quizItem) {
                    if (!$quizItem->questionItem->question->questionId) {
                        continue;
                    }
                    $this->db->insert('qz_question', [
                        'form_id' => $form->id,
                        'question_id' => $quizItem->questionItem->question->questionId,
                        'title' => $quizItem->title,
                        'pointValue' => $quizItem->questionItem->question->grading->pointValue ?? 0,
                        'added_at' => date('Y-m-d H:i:s'),
                    ]);
                }
            }
        }
        $nullForms = $this->db->where('modifiedTime', null)->get('qz_form')->result();
        foreach ($nullForms as $form) {
            $file = $drive->files->get($form->form_id, ['fields' => 'modifiedTime']);
            $this->db->where('id', $form->id)->update('qz_form', [
                'modifiedTime' => sql_date($file->modifiedTime, true),
            ]);
            $this->db->where('form_id', $form->id)->delete('qz_question');
            $quiz = $this->getFormDetails($form->form_id);
            foreach ($quiz->items as $quizItem) {
                if (!$quizItem->questionItem->question->questionId) {
                    continue;
                }
                $this->db->insert('qz_question', [
                    'form_id' => $form->id,
                    'question_id' => $quizItem->questionItem->question->questionId,
                    'title' => $quizItem->title,
                    'pointValue' => $quizItem->questionItem->question->grading->pointValue ?? 0,
                    'added_at' => date('Y-m-d H:i:s'),
                ]);
            }
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function syncResults()
    {
        $formInstance = $this->googleClient->getFormInstance();
        $forms = $this->getActiveQuiz();

        $nextPageToken = null;
        $this->db->trans_start();
        foreach ($forms as $form) {
            do {
                $optParams = array(
                    'pageSize' => 5,
                    'pageToken' => $nextPageToken,
                );
                try {
                    $responses = $formInstance->forms_responses->listFormsResponses($form->form_id, $optParams);
                } catch (\Throwable $e) {
                    $this->log(sprintf('Error while retrieving quiz (%s) results: %s', $form->form_id, $e->getMessage()));
                }
                $nextPageToken = $responses->nextPageToken;

                if ($responses->responses) {
                    foreach ($responses->responses as $response) {
                        if ($form->record_purposes == 0 && strtotime($form->added_at) > strtotime($response->createTime)) {
                            continue;
                        }
                        if (strtotime($response->createTime) > strtotime($form->due_date)) {
                            continue;
                        }
                        if (!trim($response->respondentEmail)) {
                            $this->log('Empty "respondentEmail" received. id:' . $form->id);
                            continue;
                        }
                        $emp_rec = $this->db->select('employee_id, full_name, business_email, direct_manager_email')->where('business_email', $response->respondentEmail)->get('employee_details')->row();
                        if (!$emp_rec) {
                            $this->log('No employee found for :' . $response->respondentEmail);
                            continue;
                        }
                        $emp_form_rec_exists = $this->db
                            ->where([
                                'form_id' => $form->id,
                                'employee_id' => $emp_rec->employee_id,
                            ])
                            ->group_start()
                            ->or_where('responseId', null)
                            ->or_where('responseId', $response->responseId)
                            ->group_end()
                            ->get('qz_rel_form_employee')
                            ->row();
                        $form_employee_id = @$emp_form_rec_exists->form_employee_id;
                        if ($emp_form_rec_exists) {
                            if ($emp_form_rec_exists->submittedAt == null || $emp_form_rec_exists->totalScore != $response->totalScore) {
                                $this->db->where(['form_id' => $form->id, 'employee_id' => $emp_rec->employee_id, 'responseId' => $emp_form_rec_exists->responseId])->update('qz_rel_form_employee', [
                                    'resultSent' => $form->{'autorelease-score'} == 1 ? 'queued' : 'held',
                                    'responseId' => $response->responseId,
                                    'totalScore' => $response->totalScore,
                                    'submittedAt' => sql_date($response->createTime, true),
                                ]);
                            } else {
                                continue;
                            }
                        } else {
                            $this->db->insert('qz_rel_form_employee', [
                                'form_id' => $form->id,
                                'employee_id' => $emp_rec->employee_id,
                                'email' => $emp_rec->business_email,
                                'responseId' => $response->responseId,
                                'totalScore' => $response->totalScore,
                                'submittedAt' => sql_date($response->createTime, true),
                                'resultSent' => $form->{'autorelease-score'} == 1 ? 'queued' : 'held',
                            ]);
                            $form_employee_id = $this->db->insert_id();
                        }
                        if ($response->answers) {
                            $old_q_recs = $this->db
                                ->select('id')
                                ->where(['form_employee_id' => $form_employee_id])
                                ->get('qz_employee_questions')->result();
                            if ($old_q_recs) {
                                $old_q_recs = array_column($old_q_recs, 'id');
                                $this->db->where_in('id', $old_q_recs)->delete('qz_employee_questions');
                                $this->db->where_in('qid', $old_q_recs)->delete('qz_employee_answers');
                            }
                            foreach ($response->answers as $answer) {
                                $this->db->insert('qz_employee_questions', [
                                    'form_employee_id' => $form_employee_id,
                                    'question_id' => $answer->questionId,
                                    'isCorrect' => $answer->grade->correct ? 1 : 0,
                                    'score' => $answer->grade->score ?? 0,
                                    'feedback' => $answer->grade->feedback ? $answer->grade->feedback->text : null,
                                ]);
                                $emp_answer_id = $this->db->insert_id();
                                if ($answer->textAnswers && $answer->textAnswers->answers) {
                                    foreach ($answer->textAnswers->answers as $attempted) {
                                        $this->db->insert('qz_employee_answers', [
                                            'qid' => $emp_answer_id,
                                            'answer' => $attempted->value,
                                        ]);
                                    }
                                }
                            }
                        }
                    }
                }
            } while (!is_null($nextPageToken));
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function listQuery($orWhere = [])
    {
        $query = $this->db->from('google_quiz_detail');
        if ($orWhere) {
            $query->group_start();
            foreach ($orWhere as $key => $value) {
                $query->or_like($key, $value, 'after', false);
            }
            $query->group_end();
        }
        return $query;
    }
    public function countAllRows()
    {
        return $this->db->count_all('qz_form');
    }
    public function countDatatable($orWhere)
    {
        $query = $this->listQuery($orWhere);
        return $query->count_all_results();
    }
    public function getDatatable($post, $orWhere)
    {
        $query = $this->listQuery($orWhere);
        $query->order_by($post['columns'][$post['order'][0]['column']]['data'], $post['order'][0]['dir'])
            ->limit($post['length'] > 0 ? $post['length'] : 0, $post['start']);
        return $query->get()->result_array();
    }
    public function log($message, $method = null)
    {
        if (!$method) {
            $method = debug_backtrace()[1]['function'];
        }
        $message = sprintf('%s | %s | %s', date('c'), $method, $message);
        @file_put_contents(
            $this->logpath . date('Ymd') . '.log',
            $message . "\n",
            FILE_APPEND
        );
    }
    public function delete($id)
    {
        $rec = $this->db->select('label')->where('id', $id)->get('qz_form')->row();
        $this->db->trans_start();
        $ids = array_column($this->db
                ->select('qz_employee_questions.id')
                ->join('qz_rel_form_employee', 'qz_employee_questions.form_employee_id = qz_rel_form_employee.form_employee_id', 'left')
                ->where('qz_rel_form_employee.form_id', $id)
                ->get('qz_employee_questions')
                ->result(), 'id');
        if ($ids) {
            $this->db->where_in('id', $id)->delete('qz_employee_questions');
            $this->db->where_in('qid', $ids)->delete('qz_employee_answers');
        }
        $this->db->where('form_id', $id)->delete('qz_email_queue');
        $this->db->where('form_id', $id)->delete('qz_rel_form_employee');
        $this->db->where('form_id', $id)->delete('qz_question');
        $this->db->where('id', $id)->delete('qz_form');
        $this->activity->set_url('quiz/all')->log([
            'quiz_deleted',
            $rec->label,
            $id,
        ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function getAnswerSheet($form_employee_id)
    {
        $recs = $this->db
            ->select([
                'qz_employee_answers.answer',
                'qz_employee_questions.question_id',
                'qz_employee_questions.isCorrect',
                'qz_employee_questions.score',
                'qz_employee_questions.feedback',
                'qz_question.title',
                'qz_question.pointValue',
            ])
            ->from('qz_employee_answers')
            ->join('qz_employee_questions', 'qz_employee_answers.qid = qz_employee_questions.id', 'left')
            ->join('qz_question', 'qz_employee_questions.question_id=qz_question.question_id', 'left')
            ->where(['qz_employee_questions.form_employee_id' => $form_employee_id])
            ->get()->result();
        $data = [];
        foreach ($recs as $rec) {
            if (!isset($data[$rec->question_id])) {
                $data[$rec->question_id] = [
                    'question_id' => $rec->question_id,
                    'title' => $rec->title,
                    'pointValue' => $rec->pointValue,
                    'isCorrect' => $rec->isCorrect,
                    'score' => $rec->score,
                    'feedback' => $rec->feedback,
                    'answers' => [],
                ];
            }
            $data[$rec->question_id]['answers'][] = $rec->answer;
        }
        return array_values($data);
    }
    public function sendResultEmails($form_id, $form_employee_id)
    {
        $rec = $this->db->select('label')->where('id', $form_id)->get('qz_form')->row();
        $emp = $this->db->select('employee_id')->where('form_employee_id', $form_employee_id)->get('qz_rel_form_employee')->row();
        $this->db->trans_start();
        if ($form_employee_id != 'all') {
            $this->db->where('form_employee_id', $form_employee_id);
        }
        $this->db->where('form_id', $form_id)->update('qz_rel_form_employee', [
            'resultSent' => 'queued',
        ]);

        if ($form_employee_id != 'all') {
            $this->activity->set_employee($emp->employee_id)->set_url('quiz/details/' . $form_id)->log([
                'quiz_results_sent_employee',
                $rec->label,
                $form_id,
            ]);
        } else {
            $this->activity->set_url('quiz/details/' . $form_id)->log([
                'quiz_results_sent',
                $rec->label,
                $form_id,
            ]);
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
}
