<?php
class Payslip_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model([
            'payroll/payroll_model' => 'payroll',
        ]);
    }
    private $base_path = WRITEPATH . 'payroll/slip/';
    private $images = [
        '3' => '3.jpg',
        '4' => '4.png',
    ];
    public function get_status($payroll_id)
    {
        $res = $this->db->select('GROUP_CONCAT(payroll_payslips.status) as status, GROUP_CONCAT(email_queue.status) as queue_status')
            ->where('payroll_id', $payroll_id)->group_by('payroll_id')
            ->join('email_queue', 'payroll_payslips.outbox_id=email_queue.id', 'left')
            ->get('payroll_payslips')
            ->row();
        return [
            'slip_status' => array_unique(explode(',', $res->status)),
            'email_status' => array_unique(explode(',', $res->queue_status)),
        ];
    }
    public function generate($payroll_id)
    {
        $this->db->trans_start();
        $this->db->where(['status' => 'pending', 'payroll_id' => $payroll_id])->update('payroll_payslips', [
            'status' => 'generated',
        ]);
        $payroll_rec = $this->db->select('division_id')->where('id', $payroll_id)->get('payroll')->row();
        $this->activity
            ->set_division($payroll_rec->division_id)
            ->set_payroll($payroll_id)
            ->log('payslips_generated');
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function get_slips($payroll_id)
    {
        return $this->db
            ->select('
        payroll_payslips.*,
        payroll_employee.hrm_id,
        payroll_employee.name,
        email_queue.status as outbox_status,
        email_queue.updated_at as sent_at,
        ')
            ->join('payroll_employee', 'payroll_payslips.employee_id=payroll_employee.employee_id AND payroll_payslips.payroll_id=payroll_employee.payroll_id', 'left')
            ->join('email_queue', 'payroll_payslips.outbox_id=email_queue.id', 'left')
            ->where('payroll_payslips.payroll_id', $payroll_id)
            ->get('payroll_payslips')->result();
    }
    public function get_slip($payroll_id, $employee_id)
    {
        return $this->db->where(['payroll_id' => $payroll_id, 'employee_id' => $employee_id])->get('payroll_payslips')->row();
    }
    public function process_slip($payroll_id, $employee_id, $lock = true, $save = false)
    {
        $slip = $this->get_slip($payroll_id, $employee_id);
        
        if(empty($slip)){
            $this->session->set_flashdata('payslip_error','Payslip did not generated successfully');
            redirect($_SERVER['HTTP_REFERER']);
        }

        $rec = $this->get_employee_data($payroll_id, $employee_id);

        if (!empty(trim($slip->body))) {
            $body = $slip->body;
        } else {
            $date_array = date_range($rec['start'], $rec['end'], '+1 day', 'Y-m-d');
            $day_range = count($date_array);
            $per_day_salary = $rec['gross_salary'] / $day_range;
            $period = date('M-y', strtotime($rec['start']));

            $body = file_get_contents(APPPATH . 'modules/payroll/views/partials/salaryslip.php');

            $present_days = $this->payroll->get_paid_days((object) $rec);

            $attendance_salary = $present_days * $per_day_salary;
            $gross_salary = $rec['gross_salary'] + $rec['arrears'] + $rec['advances'] + $rec['incentives'];
            $attendance_salary_after_arrears = $attendance_salary + $rec['arrears'] + $rec['advances'] + $rec['incentives'];
            
            $income_tax = calculate_income_tax($attendance_salary_after_arrears);
            $provident_fund = round($rec['provident_fund']);
            $absent_salary = round(($day_range - $present_days - $rec['unpaid_half_days']) * $per_day_salary);
            $half_day_salary = round($rec['unpaid_half_days'] * $per_day_salary);
            $late_penalty = round(floor($rec['late_arrivals'] / 3) * $per_day_salary);

            $deductions = $income_tax;
            $deductions += $provident_fund;
            $deductions += $absent_salary;
            $deductions += $rec['ncns_penalty'];
            $deductions += $half_day_salary;
            $deductions += $rec['eobi'];
            $deductions += $late_penalty;
            $deductions += $rec['others'];

            $net_salary = $gross_salary - $deductions;
            $body = $this->_process_slip($body, [
                'current' => [
                    'date' => date('d-M-y'),
                ],
                'employee' => [
                    'hrm_id' => $rec['hrm_id'],
                    'name' => $rec['name'],
                    'city' => $rec['center'],
                    'designation' => $rec['position'],
                    'department' => $rec['department'],
                    'basic_salary' => [
                        'value' => $rec['basic_salary'],
                        'type' => 'currency',
                    ],
                    'other_allowances' => [
                        'value' => $rec['other_allowances'],
                        'type' => 'currency',
                    ],
                    'provident_fund' => [
                        'value' => $rec['provident_fund'],
                        'type' => 'currency',
                    ],
                    'eobi' => [
                        'value' => $rec['eobi'],
                        'type' => 'currency',
                    ],
                ],
                'payroll' => [
                    'period' => $period,
                    'days' => $day_range,
                    'presents' => $present_days - ($rec['paid_leaves'] + $rec['special_leaves']),
                    'arrears' => [
                        'value' => $rec['arrears'] + $rec['advances'] + $rec['incentives'],
                        'type' => 'currency',
                    ],
                    'gross_salary' => [
                        'value' => $gross_salary,
                        'type' => 'currency',
                    ],
                    'paid_leaves' => $rec['paid_leaves'] + $rec['special_leaves'],
                    'suspensions' => [
                        'value' => $rec['suspensions'],
                        'type' => 'decimal',
                    ],
                    'unpaid_leaves' => [
                        'value' => $rec['actual_absents'],
                        'type' => 'decimal',
                    ],
                    'salary_absent' => [
                        'value' => $absent_salary,
                        'type' => 'currency',
                    ],
                    'income_tax' => [
                        'value' => $income_tax,
                        'type' => 'currency',
                    ],
                    'ncns_penalty' => [
                        'value' => $rec['ncns_penalty'],
                        'type' => 'currency',
                    ],
                    'salary_half_days' => [
                        'value' => $half_day_salary,
                        'type' => 'currency',
                    ],
                    'salary_late_arrivals' => [
                        'value' => $late_penalty,
                        'type' => 'currency',
                    ],
                    'other_deductions' => [
                        'value' => $rec['others'],
                        'type' => 'currency',
                    ],
                    'total_deductions' => [
                        'value' => $deductions,
                        'type' => 'currency',
                    ],
                    'net_salary' => [
                        'value' => $net_salary,
                        'type' => 'currency',
                    ],
                ],
            ]);
        }

        $this->load->library(['pdfgenerator' => 'pdf']);
        $file_name = sprintf('%s_%s.pdf', $rec['hrm_id'], $period);
        if ($save) {
            if (!is_dir($this->base_path . date('m-Y'))) {
                mkdir($this->base_path . date('m-Y'));
            }
            if (file_exists($this->base_path . date('m-Y') . '/' . $file_name)) {
                $file_name = sprintf('%s_%s-%s.pdf', $rec['hrm_id'], $period, time());
            }
            $this->db->where(['payroll_id' => $payroll_id, 'employee_id' => $employee_id])->update('payroll_payslips', [
                'body' => $body,
            ]);
        }
        $this->pdf->html_to_pdf($body, $this->base_path . date('m-Y') . '/' . $file_name, $save, WRITEPATH . 'payroll/bgs/' . $this->images[$rec['division_id']], $lock ? date('d/m/Y', strtotime($rec['dob'])) : null, null, 'A4');
        return $this->base_path . date('m-Y') . '/' . $file_name;
    }
    private function _process_slip($body, $params)
    {
        $body = preg_replace('|(<!--\s*pagebreak\s*-->.*?)\n*(<[\w\d\s]+)|', '$1<span style="page-break-after:always"></span>$2', $body);
        $body = preg_replace_callback(
            "|{{\S+?}}|",
            function ($matches) use ($params) {
                $matches_arr = strtolower(trim(trim(strip_tags($matches[0]), '{}')));
                $matches_arr = explode(':', strtolower($matches_arr));

                $matches_arr[1] = preg_replace('#[^\w\d\_]+#is', '_', $matches_arr[1]);
                $matches_arr[1] = strtolower($matches_arr[1]);

                if ($matches_arr && count($matches_arr) == 2) {
                    if (isset($params[$matches_arr[0]]) && isset($params[$matches_arr[0]][$matches_arr[1]])) {
                        $value = $params[$matches_arr[0]][$matches_arr[1]];
                        if (is_array($value)) {
                            switch ($value['type']) {
                                case 'currency':{
                                        $value['value'] = number_format($value['value'], 0);
                                        break;
                                    }
                                case 'decimal':{
                                        $value['value'] = $value['value'] + 0;
                                        break;
                                    }
                            }
                            return $value['value'];
                        } else {
                            return $value;
                        }
                    }
                }
                return $matches[0];
            },
            $body);
        return $body;
    }
    public function get_employee_data($payroll_id, $employee_id)
    {
        $employee = $this->db->query(
            'CALL get_payroll_employee(?, ?);',
            [$payroll_id, $employee_id]
        )->row_array();
        $this->db->next_result();
        $employee = decrypt_employee_data($employee, ['dob'], true, 'payroll_employee');
        $employee['actual_presents'] = $this->payroll->get_paid_days((object) $employee);
        $employee['actual_absents'] = $employee['ncns'] + $employee['unpaid_leaves'];
        $employee['other_allowances'] = percentage_to_amount($employee['gross_salary'], 32);
        $employee['provident_fund'] = round($employee['provident_fund']);
        $employee['income_tax'] = 0;
        $employee['arrears'] = $employee['arrears'] ?? 0;
        $employee['advances'] = $employee['advances'] ?? 0;
        $employee['incentives'] = $employee['incentives'] ?? 0;
        $employee['ncns_penalty'] = $employee['ncns_penalty'] ?? 0;
        $employee['others'] = $employee['others'] ?? 0;
        return $employee;
    }
    public function cleanup()
    {
        $dir = $this->base_path . date('m-Y', strtotime('-3 months'));
        if (is_dir($dir)) {
            $di = new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS);
            $ri = new RecursiveIteratorIterator($di, RecursiveIteratorIterator::CHILD_FIRST);
            foreach ($ri as $file) {
                $file->isDir() ? rmdir($file) : unlink($file);
            }
            @rmdir($dir);
        }
    }
}
