<?php

use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class Payorder_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model([
            'payroll/payroll_model' => 'payroll',
        ]);
    }
    public function get($payroll_id)
    {
        $recs = $this->db->query(
            'CALL get_payorders(?);',
            [$payroll_id]
        )->result();
        $this->db->next_result();
        return array_map(function ($rec) {
            $rec = decrypt_employee_data($rec, ['dob'], false, 'payroll_employee');
            $rec->actual_presents = $this->payroll->get_paid_days($rec);
            $rec->actual_absents = $rec->ncns + $rec->unpaid_leaves;
            $rec->arrears = $rec->arrears ?? 0;
            $rec->advances = $rec->advances ?? 0;
            $rec->incentives = $rec->incentives ?? 0;
            $rec->ncns_penalty = $rec->ncns_penalty ?? 0;
            $rec->others = $rec->others ?? 0;
            $rec->income_tax = 0;
            return $rec;
        }, $recs);
    }
    public function get_employee($payroll_id, $employee_id)
    {
        return $this->db->select('payroll_employee.*, payroll_payorders.account_title, payroll_payorders.comments')
            ->join('payroll_employee', 'payroll_payorders.payroll_id=payroll_employee.payroll_id AND payroll_payorders.employee_id=payroll_employee.employee_id', 'left')
            ->where(['payroll_payorders.payroll_id' => $payroll_id, 'payroll_payorders.employee_id' => $employee_id])
            ->get('payroll_payorders')
            ->row();
    }
    public function update($payroll_id, $employee_id, $account_title, $comments)
    {
        return $this->db->where([
            'payroll_id' => $payroll_id,
            'employee_id' => $employee_id,
        ])->update('payroll_payorders', [
            'account_title' => strip_tags($account_title),
            'comments' => strip_tags($comments),
        ]);
    }
    public function get_header()
    {
        return [
            ['S.No.', 6],
            ['Empl. ID', 10],
            ['Name of Employee', 35],
            ['Center', 12],
            ['Date of Joining', 11],
            'Department',
            'Title of Account',
            'CNIC #',
            'Net Salary',
            'Payorder Fee',
            'Net Payable',
            'Comments',
        ];
    }
    public function get_header_styles()
    {
        return [
            'fill' => [
                'color' => [
                    'rgb' => 'C5BE97',
                ],
                'fillType' => Fill::FILL_SOLID,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => [
                        'rgb' => '000000',
                    ],
                ],
            ],
            'font' => [
                'bold' => 1,
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
                'wrapText' => 1,
            ],
        ];
    }
    public function get_meta()
    {
        require_once APPPATH . 'libraries/Mysql_type.php';
        $mysql_types = new Mysql_type();
        return [
            $mysql_types::TYPE_INT24, /* S NO */
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_VARCHAR,
            $mysql_types::TYPE_INT24,
            $mysql_types::TYPE_INT24,
            $mysql_types::TYPE_INT24,
            $mysql_types::TYPE_VARCHAR,
        ];
    }
    public function create($id)
    {
        $payroll = $this->payroll->get($id);
        if (!$payroll) {
            show_404();
        }
        $data['month_info'] = month_first_last_day(date('m', strtotime($payroll->month)), $payroll->year);
        $data['date_array'] = date_range($payroll->start, $payroll->end, '+1 day', 'Y-m-d');
        $day_range = count($data['date_array']);
        $payorders = $this->get($id);

        $prepared_data = [];
        foreach ($payorders as $serial => $payorder) {
            $deductions = 0;

            $attendance_salary = round($this->payroll->calculate_attendance_salary($payorder, $day_range));

            $attendance_salary_after_arrears = $attendance_salary + $payorder->arrears + $payorder->advances + $payorder->incentives;
            $payorder->income_tax = calculate_income_tax($attendance_salary_after_arrears);
            $deductions += $payorder->income_tax;
            $deductions += $payorder->eobi;
            $deductions += round($payorder->provident_fund);
            $deductions += round($payorder->ncns_penalty);
            $deductions += round($payorder->others);
            $net_salary = $attendance_salary_after_arrears - $deductions;

            $payorder_minus_net = $net_salary + 290;
            $prepared_data[] = [
                $serial + 1,
                $payorder->hrm_id,
                $payorder->name,
                $payorder->center,
                formatted_date($payorder->hire_date),
                $payorder->department,
                $payorder->account_title,
                $payorder->cnic,
                $payorder_minus_net,
                290,
                $net_salary,
                $payorder->comments,
            ];
        }
        return $prepared_data;
    }
    public function _spreadsheet($id)
    {
        $payroll = $this->payroll->get($id);
        if (!$payroll) {
            show_404();
        }
        $spreadsheet = new Spreadsheet();
        $properties = $spreadsheet->getProperties();

        $properties->setCreator('HRM Payroll System');
        $properties->setLastModifiedBy('HRM Payroll System');
        $properties->setTitle(sprintf('Payroll-%s, %s', ucfirst($payroll->month), $payroll->year));
        $properties->setSubject(sprintf('Payroll-%s, %s', ucfirst($payroll->month), $payroll->year));
        $properties->setDescription(sprintf('Auto generated sheet for payroll of %s, %s period', ucfirst($payroll->month), $payroll->year));

        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setShowGridlines(false);
        $sheet->setTitle('Payorders');

        $header = $this->get_header();
        foreach ($header as $column => $value) {
            $column += 1;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValueExplicit(
                is_string($value) ? $value : $value[0],
                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
            );
            $sheet->getColumnDimension($cell->getColumn())->setWidth(is_string($value) ? 20 : $value[1]);
        }
        $sheet->getRowDimension(1)->setRowHeight(70);
        $sheet->getStyle('A1:L1')
            ->applyFromArray($this->get_header_styles());
        $sheet->getStyle('K1:K1')
            ->applyFromArray([
                'fill' => [
                    'color' => [
                        'argb' => '00000000',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
                'font' => [
                    'color' => [
                        'argb' => '00FFFFFF',
                    ],
                ],
            ]);
        $sheet->freezePane('D1');

        $data['month_info'] = month_first_last_day(date('m', strtotime($payroll->month)), $payroll->year);
        $data['date_array'] = date_range($payroll->start, $payroll->end, '+1 day', 'Y-m-d');
        $day_range = count($data['date_array']);
        $payorders = $this->get($id);

        $emp_no = 0;
        $row = 1;
        foreach ($payorders as $payorder) {
            $emp_no++;
            $row++;
            $deductions = 0;

            $attendance_salary = round($this->payroll->calculate_attendance_salary($payorder, $day_range));

            $attendance_salary_after_arrears = $attendance_salary + $payorder->arrears + $payorder->advances + $payorder->incentives;
            $payorder->income_tax = calculate_income_tax($attendance_salary_after_arrears);
            $deductions += $payorder->income_tax;
            $deductions += $payorder->eobi;
            $deductions += round($payorder->provident_fund);
            $deductions += round($payorder->ncns_penalty);
            $deductions += round($payorder->others);
            $net_salary = $attendance_salary_after_arrears - $deductions;

            $payorder_minus_net = $net_salary + 290;
            for ($column = 1; $column <= 12; $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;
                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $payorder->hrm_id;
                            break;
                        }
                    case 3:{
                            $value = $payorder->name;
                            break;
                        }
                    case 4:{
                            $value = $payorder->center;
                            break;
                        }
                    case 5:{
                            $value = $payorder->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            $cell->setValue($timestamp);
                            $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_XLSX14);
                            continue 2;
                            break;
                        }
                    case 6:{
                            $value = $payorder->department;
                            break;
                        }
                    case 7:{
                            $value = $payorder->account_title;
                            break;
                        }
                    case 8:{
                            $value = cnic_with_dashes($payorder->cnic);
                            break;
                        }
                    case 9:{
                            $value = (float) $payorder_minus_net;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('PKR #,##0');
                            break;
                        }
                    case 10:{
                            $value = 290;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 11:{
                            $value = (float) $net_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('PKR #,##0');
                            break;
                        }
                    case 12:{
                            $value = $payorder->comments;
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getCellByColumnAndRow(10, $row + 2)->setValue('Total')->getStyle()->applyFromArray([
            'font' => [
                'size' => 15,
                'bold' => true,
            ],
            'borders' => [
                'top' => [
                    'borderStyle' => Border::BORDER_DOUBLE,
                    'color' => [
                        'rgb' => '000000',
                    ],
                ],
                'bottom' => [
                    'borderStyle' => Border::BORDER_DOUBLE,
                    'color' => [
                        'rgb' => '000000',
                    ],
                ],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
                'wrapText' => 1,
            ],
        ]);
        $sheet->getCellByColumnAndRow(11, $row + 2)->setValue("=SUM(K2:K{$row})")->getStyle()->applyFromArray([
            'font' => [
                'size' => 15,
                'bold' => true,
            ],
            'numberFormat' => [
                'formatCode' => 'PKR #,##0',
            ],
            'borders' => [
                'top' => [
                    'borderStyle' => Border::BORDER_DOUBLE,
                    'color' => [
                        'rgb' => '000000',
                    ],
                ],
                'bottom' => [
                    'borderStyle' => Border::BORDER_DOUBLE,
                    'color' => [
                        'rgb' => '000000',
                    ],
                ],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
                'wrapText' => 1,
            ],
        ]);
        $sheet->getStyle("A2:L{$row}")
            ->applyFromArray([
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);
        $sheet->getStyle("C2:C{$row}")
            ->applyFromArray([
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_LEFT,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ]);
        $sheet->getStyle("K2:K{$row}")
            ->applyFromArray([
                'font' => [
                    'bold' => 1,
                ],
                'fill' => [
                    'color' => [
                        'rgb' => 'BFBFBF',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
            ]);
        return [
            'file_name' => sprintf('Payorders_%s_%s-%s', $payroll->division, ucfirst($payroll->month), $payroll->year),
            'spreadsheet' => $spreadsheet,
        ];
    }
    public function download($id)
    {
        $data = $this->_spreadsheet($id);
        $writer = new Xlsx($data['spreadsheet']);
        header('Content-Description: File Transfer');
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $data['file_name'] . '.xlsx"');
        header('Content-Transfer-Encoding: binary');

        $writer->save('php://output');
    }
}
