<?php defined('BASEPATH') or exit('No direct script access allowed');
class Stats_model extends CI_Model
{
    private $_flags = [
        'P' => 'present',
        'A' => 'absent',
        'LA' => 'late_arrival',
        'UH-D' => 'unpaid_half_day',
        'H-D' => 'half_day',
        'SUS' => 'suspensions',
        'PL' => 'paid_leave',
        'SPL' => 'special_paid_leave',
        'OFF' => 'off',
    ];
    private $__allow_full_monthly_stats = [
        'anniversary' => [32, 447],
        'birthday' => [32, 447],
        'joining' => [32, 447],
        'separation' => [12, 32, 447],
    ];
    public function flag_description($flag)
    {
        return slug_to_readable($this->_flags[$flag]);
    }
    public function employees()
    {
        if (!($this->session->userdata('supervisor') || $this->permission->method('manage_employee', 'update')->access())) {
            return sendJson([]);
        }
        $query = "SELECT
            SUM(IF(status = 'active', `total`, 0)) AS `active`,
            SUM(IF(status = 'on-hold', `total`, 0)) AS `on_hold`,
            SUM(IF(status = 'resignation-received',
                `total`,
                0)) AS `resignation_received`,
            SUM(IF(status NOT IN ('active' , 'resignation-received', 'on-hold'),
                `total`,
                0)) AS `others`,
            SUM(`total`) AS `total`
        FROM
            (SELECT
                `status`, COUNT(`employee_id`) AS `total`
            FROM
                `employee_history`";
        $where = [];
        if (!$this->permission->module('show_inactive_employees')->access()) {
            $where[] = "status = 'active'";
        }
        if ($this->permission->method('manage_employee', 'update')->access()) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model', 'employees_model');
            $employee_ids = $this->employees_model->get_subordinates();
            if ($employee_ids) {
                $employee_ids = implode(',', array_column($employee_ids, 'employee_id'));
                $where[] = 'employee_id IN (' . $employee_ids . ')';
            } else {
                return [
                    'total' => 0,
                ];
            }
        }
        if ($where) {
            $query .= ' WHERE ';
            $query .= implode(' AND ', $where);
        }
        $query .= " GROUP BY status) AS employees;";
        $recs = $this->db
            ->query($query)->row_array();
        return array_filter($recs);
    }
    public function today_unmarked_attendance()
    {
        $employee_id = $this->session->userdata('employee_id');
        $today = date('Y-m-d');
        $employee = $this->db->select('employee_history.employee_id, employee_history.first_name, employee_history.last_name, employee_history.hrm_id')
        ->join('attendance_history','employee_history.employee_id=attendance_history.uid')
        ->where('employee_history.employee_id',$employee_id)
        ->where(['attendance_history.uid' => $employee_id, 'attendance_history.day' => $today, 'attendance_history.status' => 'in'])->get('employee_history')->result();
        return $employee;
    }
    public function attendance($date = null, $get_employees = false)
    {
        if (!$date) {
            $date = date('Y-m-d H:i:s');
        }
        $working_day = current_working_day($date);
        $working_day = date('Y-m-d H:i:s', strtotime($working_day . ' -1 hour'));
        $departments = $this->get_departments();
        $suspensions = $this->get_suspensions($working_day);
        $weekends = get_weekend();
        $holidays = $this->get_holiday($working_day);
        $leaves = $this->get_leaves($working_day);
        $leave_types = $this->get_leave_types();

        $attendance_filtered = [];

        $employee_ids = [];
        $exempted_employee_ids = $this->get_exempted();
        if ($this->permission->method('manage_employee', 'update')->access()) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model', 'employees_model');
            $employee_ids = $this->employees_model->get_subordinates();
            if (!$employee_ids) {
                return [
                    'summary' => [],
                    'leave_types' => [],
                    'departments' => [],
                ];
            }
            $employee_ids = array_column($employee_ids, 'employee_id');
        }
        $query = "SELECT
        GROUP_CONCAT(IF(attendance_view.flag = '' AND attendance_view.punchout_time IS NULL, 'P', attendance_view.flag)) as flags,
        employee_history.employee_id as employee_id,
        employee_history.dept_id as dept_id,
        employee_history.shift as shift
        FROM attendance_view
        LEFT JOIN employee_history ON attendance_view.uid = employee_history.employee_id
        WHERE employee_history.status IN ('active', 'resignation-received')
        AND attendance_view.day = DATE(?)";

        if (!$this->permission->module('show_inactive_employees')->access()) {
            $query .= " AND employee_history.status = 'active'";
        }
        if ($employee_ids) {
            $query .= " AND attendance_view.uid IN (" . implode(',', $employee_ids) . ")";
        }
        if ($exempted_employee_ids) {
            $query .= " AND attendance_view.uid NOT IN (" . implode(',', $exempted_employee_ids) . ")";
        }
        $query .= " GROUP BY attendance_view.uid";
        $attendance = $this->db
            ->query($query,
                [$working_day])->result_array();

        $this->db->select('employee_id, \'\' as flags, 0 as total, dept_id, shift')->where_in('status', ['active', 'resignation_received']);
        if ($attendance) {
            $this->db->where_not_in('employee_id', array_column($attendance, 'employee_id'));
        }
        if ($exempted_employee_ids) {
            $this->db->where_not_in('employee_id', $exempted_employee_ids);
        }
        if ($employee_ids) {
            $this->db->where_in('employee_id', $employee_ids);
        }
        $absent_employees = $this->db->get('employee_history')->result_array();
        $attendance = array_merge($attendance, $absent_employees);

        foreach ($attendance as $atn) {
            $atn['flags'] = array_filter(explode(',', $atn['flags']));
            $atn['flags'] = array_unique($atn['flags']);
            if (in_array($atn['employee_id'], $suspensions)) {
                $attendance_filtered[] = [
                    'employee_id' => $atn['employee_id'],
                    'punches' => $atn['total'],
                    'leave_type_id' => $leaves[$atn['employee_id']]['leave_type_id'],
                    'flag' => 'SUS',
                ];
                $departments[$atn['dept_id']]['attendance'][$this->_flags['SUS']]++;
                continue;
            }
            if ($leaves[$atn['employee_id']] && $leaves[$atn['employee_id']]['type'] != 'half_day') {
                $departments[$atn['dept_id']]['attendance'][$this->_flags['PL']]++;
                $leave_types[$leaves[$atn['employee_id']]['leave_type_id']]['count']++;

                $attendance_filtered[] = [
                    'employee_id' => $atn['employee_id'],
                    'punches' => $atn['total'],
                    'leave_type_id' => $leaves[$atn['employee_id']]['leave_type_id'],
                    'flag' => 'PL',
                ];
                continue;
            }
            if ($leaves[$atn['employee_id']] && $leaves[$atn['employee_id']]['type'] == 'half_day') {
                if (in_array('FD', $atn['flags']) || in_array('OT', $atn['flags']) || in_array('UH-D', $atn['flags'])) {
                    $flag = 'P';
                    $departments[$atn['dept_id']]['attendance'][$this->_flags['P']]++;
                } elseif (in_array('A', $atn['flags']) || !$atn['flags']) {
                    $flag = 'H-D';
                    $departments[$atn['dept_id']]['attendance'][$this->_flags['H-D']]++;
                }
                $attendance_filtered[] = [
                    'employee_id' => $atn['employee_id'],
                    'punches' => $atn['total'],
                    'leave_type_id' => $leaves[$atn['employee_id']]['leave_type_id'],
                    'flag' => $flag,
                ];
                continue;
            }
            $flag = null;
            if ($atn['flags']) {
                if (in_array('FD', $atn['flags']) || in_array('OT', $atn['flags'])) {
                    $flag = 'P';
                } elseif (in_array('UH-D', $atn['flags'])) {
                    $flag = 'UH-D';
                } elseif (in_array('A', $atn['flags']) && ($atn['total'] % 2 != 0)) {
                    $flag = 'A';
                } elseif (in_array('LA', $atn['flags'])) {
                    $flag = 'LA';
                } elseif (in_array('P', $atn['flags'])) {
                    $flag = 'P';
                }
                if (($flag == 'P' || !$flag) && in_array('LA', $atn['flags'])) {
                    $flag = 'LA';
                }
                $departments[$atn['dept_id']]['attendance'][$this->_flags[$flag]]++;
            } else {
                if ($holidays[$atn['shift']]) {
                    $flag = 'OFF';
                    $departments[$atn['dept_id']]['attendance'][$this->_flags['OFF']]++;
                } elseif (in_array(date('l', strtotime($working_day)), $weekends[$atn['shift']] ?? [])) {
                    $flag = 'OFF';
                    $departments[$atn['dept_id']]['attendance'][$this->_flags[$flag]]++;
                } else {
                    $flag = 'A';
                    $departments[$atn['dept_id']]['attendance'][$this->_flags[$flag]]++;
                }
            }
            $attendance_filtered[] = [
                'employee_id' => $atn['employee_id'],
                'punches' => $atn['total'],
                'flag' => $flag,
            ];
        }
        if ($get_employees) {
            return $attendance_filtered;
        }
        $raw_flags = array_column($attendance_filtered, 'flag');
        $counts_raw = array_count_values($raw_flags);
        $count_labels = array_unique($raw_flags);

        $count_wfh = $this->db->where(['work_from_home' => 1, 'status' => 'active'])->get('employee_history')->num_rows();
        $counts = [
            'present' => 0,
            'absent' => 0,
            'late_arrival' => 0,
            'unpaid_half_day' => 0,
            'half_day' => 0,
            'suspensions' => 0,
            'paid_leave' => 0,
            'off' => 0,
            'work_from_home' => $count_wfh,
        ];
        foreach ($count_labels as $count_label) {
            $counts[$this->_flags[$count_label] ?? $count_label] = $counts_raw[$count_label] ? $counts_raw[$count_label] : 0;
        }
        $counts['hidden'] = $this->db->where('user_id', $this->session->userdata('id'))->count_all_results('attendance_stat_exempt');

        $departments = array_values($departments);
        $leave_types = array_values($leave_types);
        $leave_types = array_filter($leave_types, function ($v) {
            return $v['count'] > 0;
        });
        return [
            'summary' => $counts,
            'leave_types' => $leave_types,
            'departments' => $departments,
        ];
    }
    public function get_suspensions($day = null)
    {
        if (!$day) {
            $day = date('Y-m-d');
        } else {
            $day = date('Y-m-d', strtotime($day));
        }

        $cache_flag = 'stats.attendance.suspensions.' . $day;
        if (!$data = $this->cache->get($cache_flag)) {
            $recs = $this->db->select('employee_penalties.employee_id')
                ->join('employee_history', 'employee_penalties.employee_id = employee_history.employee_id', 'left')
                ->where_in('employee_history.status', ['active', 'resignation_received'])
                ->where([
                    'employee_penalties.type' => 'suspension',
                    'employee_penalties.date' => $day,
                ])->get('employee_penalties')->result();
            $data = array_column($recs, 'employee_id');
            $this->cache->save($cache_flag, $data, 120);
        }
        return $data;
    }
    public function get_departments()
    {
        $cache_flag = 'stats.attendance.departments';
        if (!$data = $this->cache->get($cache_flag)) {
            $recs = $this->db->select('dept_id as id, department_name as name')->order_by('department_name', 'asc')->get('department')->result();
            $data = [];
            foreach ($recs as $rec) {
                $data[$rec->id] = [
                    'label' => $rec->name,
                    'attendance' => [
                        'present' => 0,
                        'absent' => 0,
                        'late_arrival' => 0,
                        'unpaid_half_day' => 0,
                        'half_day' => 0,
                        'suspensions' => 0,
                        'paid_leave' => 0,
                        'off' => 0,
                    ],
                ];
            }
            $this->cache->save($cache_flag, $data, 120);
        }
        return $data;
    }
    public function get_leaves($day = null)
    {
        if (!$day) {
            $day = date('Y-m-d');
        } else {
            $day = date('Y-m-d', strtotime($day));
        }

        $cache_flag = 'stats.attendance.get_leaves.' . $day;
        if (!$data = $this->cache->get($cache_flag)) {
            $recs = $this->db
                ->query("SELECT leave_apply.employee_id, leave_apply.leave_type_id, employee_history.dept_id, leave_applied_days.type
            FROM leave_apply
            LEFT JOIN leave_applied_days ON leave_apply.leave_appl_id = leave_applied_days.leave_appl_id AND leave_applied_days.day = ? AND leave_applied_days.active = 1
            LEFT JOIN leave_type ON leave_apply.leave_type_id = leave_type.leave_type_id
            LEFT JOIN employee_history ON leave_apply.employee_id = employee_history.employee_id
            WHERE hr_status = 'approved' AND leave_aprv_strt_date <= ? AND leave_aprv_end_date >= ?",
                    [$day, $day, $day])->result_array();

            $data = [];
            foreach ($recs as $rec) {
                $data[$rec['employee_id']] = $rec;
            }
            $this->cache->save($cache_flag, $data, 120);
            return $data;
        }
        return $data;
    }
    public function get_leave_types()
    {
        $cache_flag = 'stats.attendance.get_leave_types';

        if (!$data = $this->cache->get($cache_flag)) {
            $recs = $this->db->select('leave_type_id as id, leave_type as name')->get('leave_type')->result_array();

            $data = [];
            foreach ($recs as $rec) {
                $data[$rec['id']] = [
                    'id' => $rec['id'],
                    'label' => $rec['name'],
                    'count' => 0,
                ];
            }
            $this->cache->save($cache_flag, $data, 300);
            return $data;
        }
        return $data;
    }
    public function get_holiday($day = null)
    {
        if (!$day) {
            $day = date('Y-m-d');
        } else {
            $day = date('Y-m-d', strtotime($day));
        }

        $cache_flag = 'stats.attendance.get_holiday.' . $day;

        if (!$data = $this->cache->get($cache_flag)) {
            $recs = $this->db->select('payroll_holiday_shift_wise.shift_id as id, payroll_holiday.holiday_name as label')
                ->from('payroll_holiday_shift_wise')
                ->join('payroll_holiday', 'payroll_holiday_shift_wise.payrl_holi_id = payroll_holiday.payrl_holi_id', 'left')
                ->where([
                    'payroll_holiday.start_date <=' => $day,
                    'payroll_holiday.end_date >=' => $day,
                ])->get()->result();

            $data = [];
            foreach ($recs as $rec) {
                $data[$rec->id] = $rec->label;
            }

            $this->cache->save($cache_flag, $data, 120);
            return $data;
        }

        return $data;
    }
    public function month_joinings($date = null, $type = 'rows', $date_filter = null)
    {
        $interval = null;
        if ($date) {
            $interval = month_first_last_day(date('m', strtotime($date)), date('Y', strtotime($date)));
        }
        else if ($date_filter) {
            $interval = month_first_last_day(date('m', strtotime($date_filter)), date('Y', strtotime($date_filter)));
        }
         else {
            $interval = month_first_last_day();
        }
        if ($this->permission->method('manage_employee', 'update')->access() || in_array($this->session->userdata('employee_id'), $this->__allow_full_monthly_stats['joining'])) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model');
            $employee_ids = $this->employees_model->get_subordinates(null, false);
            $employee_ids = array_column($employee_ids, 'employee_id');
            if ($employee_ids) {
                $this->db->where_in('employee_id', $employee_ids);
            } else {
                return $type == 'count' ? 0 : [];
            }
        } else {
            return $type == 'count' ? 0 : [];
        }
        $this->db->select('employee_id, hrm_id, first_name, last_name, status, hire_date, direct_manager_name, division, department, position')
            ->where([
                'hire_date >=' => $interval['first'],
                'hire_date <=' => $interval['last'],
                'termination_date' => null,
            ])
            ->order_by('hire_date')
            ->from('employee_details');
        switch ($type) {
            case 'count':
                return $this->db->count_all_results();
                break;
            default:
                return $this->db->get()->result();
                break;
        }
    }
    public function month_resignations($date = null, $type = 'rows', $date_filter = null)
    {
        $interval = null;
        if ($date) {
            $interval = month_first_last_day(date('m', strtotime($date)), date('Y', strtotime($date)));
        }
        else if ($date_filter) {
            $interval = month_first_last_day(date('m', strtotime($date_filter)), date('Y', strtotime($date_filter)));
        } 
        else {
            $interval = month_first_last_day();
        }
        if ($this->permission->method('manage_employee', 'update')->access() || in_array($this->session->userdata('employee_id'), $this->__allow_full_monthly_stats['separation'])) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model');
            $employee_ids = $this->employees_model->get_subordinates(null, false, true);
            $employee_ids = array_column($employee_ids, 'employee_id');
            if ($employee_ids) {
                $this->db->where_in('employee_id', $employee_ids);
            } else {
                return $type == 'count' ? 0 : [];
            }
        } else {
            return $type == 'count' ? 0 : [];
        }
        $this->db->select('employee_id, hrm_id, first_name, last_name, status, termination_date, direct_manager_name, division, department, position')
            ->where([
                'termination_date >' => $interval['first'],
                'termination_date <=' => date('Y-m-d', strtotime($interval['last'] . ' +1 day')),
            ])
            ->order_by('termination_date')
            ->from('employee_details');
        switch ($type) {
            case 'count':
                return $this->db->count_all_results();
                break;
            default:
                return $this->db->get()->result();
                break;
        }
    }
    public function month_anniversaries($date = null, $type = 'rows', $date_filter = null)
    {
        $interval = null;
        if ($date) {
            $interval = month_first_last_day(date('m', strtotime($date)), date('Y', strtotime($date)));
        } 
        else if ($date_filter) {
            $interval = month_first_last_day(date('m', strtotime($date_filter)), date('Y', strtotime($date_filter)));
        } 
        else {
            $interval = month_first_last_day();
        }
        if ($this->permission->method('manage_employee', 'update')->access() || in_array($this->session->userdata('employee_id'), $this->__allow_full_monthly_stats['anniversary'])) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model');
            $employee_ids = $this->employees_model->get_subordinates(null, false, true);
            $employee_ids = array_column($employee_ids, 'employee_id');
            if ($employee_ids) {
                $this->db->where_in('employee_id', $employee_ids);
            } else {
                return $type == 'count' ? 0 : [];
            }
        } else {
            return $type == 'count' ? 0 : [];
        }
        $this->db->select('employee_id, hrm_id, first_name, last_name, status, direct_manager_name, division, department, position, hire_date, tenure')
            ->where([
                'employee_details.status' => 'active',
                'YEAR(employee_details.hire_date) <' => date('Y', strtotime($interval['first'])),
                "DATE_FORMAT(employee_details.hire_date,'%m') =" => date('m', strtotime($interval['first'])),
                "DATE_FORMAT(employee_details.hire_date,'%d') >=" => 1,
                "DATE_FORMAT(employee_details.hire_date,'%d') <=" => date('d', strtotime($interval['last'])),
            ])
            ->order_by("DATE_FORMAT(employee_details.hire_date, '%m-%d')")
            ->from('employee_details');
        switch ($type) {
            case 'count':
                return $this->db->count_all_results();
                break;
            default:
                return $this->db->get()->result();
                break;
        }
    }
    public function month_birthdays($date = null, $type = 'rows', $date_filter = null)
    {
        $interval = null;
        if ($date) {
            $interval = month_first_last_day(date('m', strtotime($date)), date('Y', strtotime($date)));
        }
        else if ($date_filter) {
            $interval = month_first_last_day(date('m', strtotime($date_filter)), date('Y', strtotime($date_filter)));
        } else {
            $interval = month_first_last_day();
        }
        if ($this->permission->method('manage_employee', 'update')->access() || in_array($this->session->userdata('employee_id'), $this->__allow_full_monthly_stats['birthday'])) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model');
            $employee_ids = $this->employees_model->get_subordinates(null, false, true);
            $employee_ids = array_column($employee_ids, 'employee_id');
            if ($employee_ids) {
                $this->db->where_in('employee_id', $employee_ids);
            } else {
                return $type == 'count' ? 0 : [];
            }
        } else {
            return $type == 'count' ? 0 : [];
        }
        $recs = $this->db
            ->select('employee_id, dob')
            ->where('status', 'active')
            ->get('employee_history')
            ->result();
        $interval['first'] = strtotime($interval['first']);
        $interval['last'] = strtotime($interval['last']);
        $year = date('Y', $interval['first']);
        $recs = array_filter($recs, function ($rec) use ($interval, $year) {
            $rec = decrypt_employee_data($rec);
            $rec->dob = strtotime(date($year . '-m-d', strtotime($rec->dob)));
            return ($rec->dob >= $interval['first'] && $rec->dob <= $interval['last']);
        });
        $employee_ids = array_column($recs, 'employee_id');
        $this->db->select('employee_id, hrm_id, first_name, last_name, status, direct_manager_name, division, department, position, dob')
            ->where_in('employee_id', $employee_ids)
            ->from('employee_details');
        switch ($type) {
            case 'count':
                if (!$employee_ids) {
                    $this->db->reset_query();
                    return 0;
                }
                return $this->db->count_all_results();
                break;
            default:
                if (!$employee_ids) {
                    $this->db->reset_query();
                    return [];
                }
                $recs = $this->db->get()->result();
                $recs = array_map(function ($rec) use ($year) {
                    $rec = decrypt_employee_data($rec);
                    $rec->dob = strtotime(date($year . '-m-d', strtotime($rec->dob)));
                    return $rec;
                }, $recs);
                usort($recs, function ($a, $b) {
                    return $a->dob <=> $b->dob;
                });
                return $recs;
                break;
        }
    }
    public function this_month($date_filter)
    {
        return [
            'anniversaries' => $this->month_anniversaries(null, 'count', $date_filter),
            'birthdays' => $this->month_birthdays(null, 'count', $date_filter),
            'joinings' => $this->month_joinings(null, 'count', $date_filter),
            'separations' => $this->month_resignations(null, 'count', $date_filter),
        ];
    }
    public function details($type, $key, $datefilter)
    {
        if ($type == 'leave') {
            return $this->leave_query($key);
        }
        if ($key == 'work_from_home') {
            return $this->work_from_home_employees();
        }
        if (!method_exists(Stats_model::class, "d_{$type}_{$key}")) {
            return ['error' => 'Method not found'];
        }
        if ($type == 'this_month') {
            return $this->{"d_{$type}_{$key}"}($datefilter);
        }
        return $this->{"d_{$type}_{$key}"}();
    }
    
    private function work_from_home_employees()
    {
        $this->db->where(['work_from_home' => 1, 'status' => 'active']);
        $recs = $this->employee_status_query();
        return ['success' => [
            'title' => 'Work From Home Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    ucfirst($rec->status),
                ];
            }, $recs),
        ]];
    }
    private function d_this_month_joinings($datefilter)
    {
        $recs = $this->month_joinings(null, 'row', $datefilter);
        $data = [
            'title' => 'This Month\'s Joinings',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status', 'Date of Joining'],
            'sortBy' => '7',
            'sortOrder' => 'asc',
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    slug_to_readable($rec->status),
                    formatted_date($rec->hire_date),
                ];
            }, $recs),
        ];
        return ['success' => $data];
    }
    private function d_this_month_separations($datefilter)
    {
        $recs = $this->month_resignations(null, 'row', $datefilter);
        $data = [
            'title' => 'This Month\'s Separations',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status', 'Separation Date'],
            'sortBy' => '7',
            'sortOrder' => 'asc',
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    slug_to_readable($rec->status),
                    formatted_date($rec->termination_date),
                ];
            }, $recs),
        ];
        return ['success' => $data];
    }
    private function d_this_month_anniversaries($datefilter)
    {
        $recs = $this->month_anniversaries(null, 'row', $datefilter);
        $data = [
            'title' => 'This Month\'s Anniversaries',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Date of Joining', 'Tenure'],
            'sortBy' => '6',
            'sortOrder' => 'asc',
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    formatted_date($rec->hire_date),
                    sprintf('%s year%s', $rec->tenure, $rec->tenure > 1 ? 's' : null),
                ];
            }, $recs),
        ];
        return ['success' => $data];
    }
    private function d_this_month_birthdays($datefilter)
    {
        $recs = $this->month_birthdays(null, 'row', $datefilter);
        $data = [
            'title' => 'This Month\'s Birthdays',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Birthday'],
            'sortBy' => '6',
            'sortOrder' => 'asc',
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    date('d F', $rec->dob),
                ];
            }, $recs),
        ];
        return ['success' => $data];
    }
    private function employee_status_query()
    {
        if (!($this->session->userdata('supervisor') || $this->permission->method('manage_employee', 'update')->access())) {
            return sendJson(['error' => 'You are not allowed to perform this action']);
        }
        if (!$this->permission->module('show_inactive_employees')->access()) {
            $this->db->where('status', 'active');
        }
        if ($this->permission->method('manage_employee', 'update')->access()) {
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model', 'employees_model');
            $employee_ids = $this->employees_model->get_subordinates();
            if ($employee_ids) {
                $employee_ids = array_column($employee_ids, 'employee_id');
                $this->db->where_in('employee_id', $employee_ids);
            } else {
                $this->db->reset_query();
                return [];
            }
        }
        return $this->db->select('employee_id, hrm_id, first_name, last_name, status, direct_manager_name, division, department, position, work_from_home')
            ->order_by('hrm_id')
            ->from('employee_details')->get()->result();
    }
    private function d_employee_active()
    {
        $this->db->where('status', 'active');
        $recs = $this->employee_status_query();

        return ['success' => [
            'title' => 'Active Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    slug_to_readable($rec->status),
                ];
            }, $recs),
        ]];
    }
    private function d_employee_on_hold()
    {
        $this->db->where('status', 'on-hold');
        $recs = $this->employee_status_query();

        return ['success' => [
            'title' => 'Employees On Hold',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    slug_to_readable($rec->status),
                ];
            }, $recs),
        ]];
    }
    private function d_employee_resignation_received()
    {
        $this->db->where('status', 'resignation-received');
        $recs = $this->employee_status_query();

        return ['success' => [
            'title' => 'Resignation Received of Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    slug_to_readable($rec->status),
                ];
            }, $recs),
        ]];
    }
    private function d_employee_others()
    {
        $this->db->where_not_in('status', ['active', 'on-hold', 'resignation-received']);
        $recs = $this->employee_status_query();

        return ['success' => [
            'title' => 'Other Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Status'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    slug_to_readable($rec->status),
                ];
            }, $recs),
        ]];
    }
    public function attendance_query($flag)
    {
        $recs = $this->attendance(null, true);
        $recs = array_filter($recs, function ($rec) use ($flag) {
            return $rec['flag'] == $flag;
        });
        $recs = array_column($recs, 'employee_id');
        return $this->db->select('employee_id, hrm_id, first_name, last_name, direct_manager_name, division, department, position, shift')
            ->where_in('employee_id', $recs)
            ->order_by('hire_date')
            ->from('employee_details')->get()->result();
    }
    private function d_attendance_present()
    {
        $recs = $this->attendance_query('P');
        return ['success' => [
            'title' => 'Present Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['P']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_absent()
    {
        $recs = $this->attendance_query('A');
        return ['success' => [
            'title' => 'Absent Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['A']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_late_arrival()
    {
        $recs = $this->attendance_query('LA');
        return ['success' => [
            'title' => 'Late Arrival Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['LA']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_unpaid_half_day()
    {
        $recs = $this->attendance_query('UH-D');
        return ['success' => [
            'title' => 'Unpaid-Half Day Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['UH-D']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_half_day()
    {
        $recs = $this->attendance_query('H-D');
        return ['success' => [
            'title' => 'Half-Day Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['H-D']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_suspensions()
    {
        $recs = $this->attendance_query('SUS');
        return ['success' => [
            'title' => 'Suspended Employees',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['SUS']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_paid_leave()
    {
        $recs = $this->attendance_query('PL');
        return ['success' => [
            'title' => 'Employees on Paid Leave',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['PL']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_off()
    {
        $recs = $this->attendance_query('OFF');
        return ['success' => [
            'title' => 'Employees with Off',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift', 'Attendance'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                    slug_to_readable($this->_flags['OFF']),
                ];
            }, $recs),
        ]];
    }
    private function d_attendance_hidden()
    {
        $recs = $this->db
            ->select('employee_details.*')
            ->from('attendance_stat_exempt')
            ->join('employee_details', 'attendance_stat_exempt.employee_id=employee_details.employee_id', 'left')
            ->where('attendance_stat_exempt.user_id', $this->session->userdata('id'))
            ->get()
            ->result();
        return ['success' => [
            'title' => 'Employees Attendance Hidden For Me',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Shift'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->shift,
                ];
            }, $recs),
        ]];
    }
    public function leave_query($type = null)
    {
        $recs = $this->attendance(null, true);
        if ($type != 'all') {
            $recs = array_filter($recs, function ($rec) use ($type) {
                return $rec['leave_type_id'] == $type;
            });
        }
        $working_day = current_working_day(null, false);
        $recs = array_column($recs, 'employee_id');
        $recs = $this->db->select('employee_details.employee_id, employee_details.hrm_id, employee_details.first_name, employee_details.last_name, employee_details.direct_manager_name, employee_details.division, employee_details.department, employee_details.position, leave_type.leave_type, leave_apply.num_aprv_day, leave_apply.leave_aprv_end_date')
            ->join('employee_details', 'leave_apply.employee_id = employee_details.employee_id', 'left')
            ->join('leave_type', 'leave_apply.leave_type_id = leave_type.leave_type_id', 'left')
            ->where([
                'hr_status' => 'approved',
                'leave_apply.leave_aprv_strt_date <= ' => $working_day,
                'leave_apply.leave_aprv_end_date >= ' => $working_day,
            ])
            ->where_in('leave_apply.employee_id', $recs)
            ->order_by('leave_apply.leave_aprv_end_date')
            ->from('leave_apply')->get()->result();
        return ['success' => [
            'title' => 'Employees on Leave',
            'headings' => ['Employee ID', 'Employee Name', 'Manager Name', 'Division', 'Department', 'Designation', 'Leave Type', 'Leave Days', 'Leave Ends'],
            'items' => array_map(function ($rec) {
                return [
                    $rec->hrm_id,
                    $rec->first_name . ' ' . $rec->last_name,
                    $rec->direct_manager_name,
                    $rec->division,
                    $rec->department,
                    $rec->position,
                    $rec->leave_type,
                    $rec->num_aprv_day == 0.5 ? 'Half Day' : sprintf('%g', $rec->num_aprv_day),
                    formatted_date($rec->leave_aprv_end_date),
                ];
            }, $recs),
        ]];
    }
    public function get_attrition_rate()
    {
        if (!$this->permission->method('manage_employee', 'update')->access()) {
            return -1;
        }
        $interval = month_first_last_day();
        $row = $this->db->query(
            'CALL get_attrition_rate(?, ?);',
            [$interval['first'], $interval['last']]
        )->row();
        $this->db->next_result();
        return $row->attrition_rate;
    }
    public function attendance_exempt($key, $hrm_id)
    {
        $employee = $this->db->select('employee_id')->where('hrm_id', $hrm_id)->get('employee_history')->row();
        if (!$employee) {
            return false;
        }
        if ($key == 'hidden') {
            return $this->db->where(['user_id' => $this->session->userdata('id'), 'employee_id' => $employee->employee_id])->delete('attendance_stat_exempt');
        } else {
            if ($this->db->where(['user_id' => $this->session->userdata('id'), 'employee_id' => $employee->employee_id])->count_all_results('attendance_stat_exempt') < 1) {
                return $this->db->insert('attendance_stat_exempt', [
                    'user_id' => $this->session->userdata('id'),
                    'employee_id' => $employee->employee_id,
                ]);
            }
        }
        return false;
    }
    public function get_exempted()
    {
        $recs = null;
        if ($this->permission->method('manage_employee', 'update')->access()) {
            $recs = $this->db->select('employee_id')->where('user_id', $this->session->userdata('id'))->get('attendance_stat_exempt')->result_array();
        } elseif ($this->session->userdata('supervisor')) {
            $this->load->model('employee/employees_model', 'employees_model');
            $employee_ids = $this->employees_model->get_subordinates();
            if (!$employee_ids) {
                return [];
            }
            $employee_ids = array_column($employee_ids, 'employee_id');
            $recs = $this->db->select('employee_id')->where('user_id', $this->session->userdata('id'))->where_in('employee_id', $employee_ids)->get('attendance_stat_exempt')->result_array();
        }
        if ($recs) {
            return array_column($recs, 'employee_id');
        }
        return [];
    }

    public function employee_attendance_query($flag,$page, $limit = 10, $type = 'rows')
    {
        $page *= $limit;
        $recs = $this->attendance(null, true);
        $recs = array_filter($recs, function ($rec) use ($flag) {

            return $rec['flag'] == $flag || ($rec['flag'] == 'LA' && $flag == 'P');
        });
        $recs = array_column($recs, 'employee_id');

        if (count($recs) == 0) {
            return [];
        }
        // $query = $this->db->select('e.employee_id, e.hrm_id, e.first_name, e.last_name, e.picture, e.hire_date, s.status')
        //     ->from('employee_details e')
        //     ->join('attendance_status s', 's.employee_id = e.employee_id AND s.day = '.date("Y-m-d").'','left')
        //     ->where_in('e.employee_id', $recs)
        //     ->order_by('e.hire_date')
        //     ->limit($limit, $page)
        //     ;
        $query = $this->db->select('e.employee_id, e.hrm_id, e.status, e.first_name, e.last_name,e.picture_thumbnail, e.hire_date, s.status, s.day')
        ->from('employee_details AS e')
        ->where_in('e.employee_id', $recs)
        ->join('attendance_status AS s', 's.employee_id = e.employee_id AND s.day = "'.date('Y-m-d').'"', 'left')
        ->order_by('e.hire_date')
        ->limit($limit, $page);
            if($type == 'count') {
                return $query->count_all_results();
            } else {
                $data = [];
                $query =  $query->get()->result();
                foreach ($query as $key => $value) {
                    if (file_exists($value->picture_thumbnail))
                    {
                        $data[]= $value;
                        // $data[$date->format('Y-m-d')][] = $value;
                    }
                    else {
                        $value->picture_thumbnail = './assets/img/icons/user.png';
                        $data[]= $value;
                    }
                }
                return $data;
            }
    }

    public function employees_leave_query($day,$page, $limit = 10, $leave_type = null)
    {
        $page *= $limit;
        $recs = $this->attendance(null, true);

        $start = date('Y-m-d', strtotime("+2 days"));
        $end =date('Y-m-d', strtotime("+1 Month"));

        $recs = array_column($recs, 'employee_id');
        $query = $this->db->select('employee_details.employee_id, employee_details.hrm_id,employee_details.picture, employee_details.first_name, employee_details.last_name,leave_applied_days.day, leave_applied_days.type')
            ->from('leave_apply')
            ->where(['leave_apply.hr_status = approved OR leave_apply.supervisor_status = approved'])
            ->join('employee_details', 'leave_apply.employee_id = employee_details.employee_id')
            ->join('leave_applied_days','leave_applied_days.leave_appl_id = leave_apply.leave_appl_id')
            ->where(['leave_applied_days.active' => 1, 'leave_applied_days.type' => 'full_day']);

            if ($leave_type == 'this_week') {
                $query = $query->where([
                    'leave_applied_days.day >= ' => $start,
                    'leave_applied_days.day <= ' => $end,
                ]);
            }
            else {
                $working_day = current_working_day($day, false);
                $query = $query->where([
                    'leave_applied_days.day = ' => $working_day,
                    'leave_apply.leave_aprv_strt_date <= ' => $working_day,
                    'leave_apply.leave_aprv_end_date >= ' => $working_day,
                ]);
            }
            $query = $query->where_in('leave_apply.employee_id', $recs)
            ->order_by('leave_apply.leave_aprv_end_date')
            ->limit($limit, $page)
            ->get()->result();
            if ($leave_type == 'this_week') {

                $startDate = new DateTime($start); 
                $endDate = new DateTime($end);
                $data = [];
                for ($date = $startDate; $date <= $endDate; $date->modify('+1 day')) {
                    foreach ($query as $key => $value) {
                        if ($date->format('Y-m-d') == $value->day) {
                            foreach ($query as $key => $value) {
                                if (file_exists($value->picture))
                                {
                                    $data[$date->format('Y-m-d')][] = $value;
                                }
                                else {
                                    $value->picture = './assets/img/icons/user.png';
                                    $data[$date->format('Y-m-d')][] = $value;
                                }
                            }
                        }
                    }
                }
                return $data;
            }
            else {
                
                $data = [];
                foreach ($query as $key => $value) {
                    if (file_exists($value->picture))
                    {
                        $data[]= $value;
                    }
                    else {
                        $value->picture = './assets/img/icons/user.png';
                        $data[]= $value;
                    }
                }
                return $data;
            }
    }
    public function break_exceeeded_emps($page, $limit = 10, $leave_type = null)
    {
        $page *= $limit;
        $day = date('Y-m-d');
        
        $recs = $this->attendance(null, true);
        $recs = array_column($recs, 'employee_id');
    
        $query = $this->db->select('e.employee_id, e.hrm_id, e.status, e.first_name, e.last_name, e.picture, e.hire_date, SUM(b.duration) as duration')
        ->from('employee_details e')
        ->where_in('e.employee_id', $recs)
        ->join('break_time AS b', 'b.employee_id = e.employee_id AND b.status = "end" AND b.date = "'.$day.'"')
        ->where_not_in('b.break_type',['washroom']);
        $query = $query
        ->group_by("b.employee_id")
        ->limit($limit, $page)->get()->result();
        $data = [];
        foreach ($query as $key => $value) {
            if (file_exists($value->picture))
            {
                $data[]= $value;
                // $data[$date->format('Y-m-d')][] = $value;
            }
            else {
                $value->picture = './assets/img/icons/user.png';
                $data[]= $value;
            }
        }
        return $data;
    }
}
