<?php

use Google\Service\Drive\DriveFile;
use GuzzleHttp\Psr7\Stream;

class HRM_Drive implements HRMFileHandlerInterface
{
    private $drive = null;
    private $CI = null;
    private const _fileParams = [
        'id',
        'name',
        'mimeType',
        'fileExtension',
        'size',
        'trashed',
        'parents',
        'createdTime',
        'modifiedTime',
    ];
    private $root_id = null;
    private $hrm_root_id = null;
    public function __construct(Google\Service\Drive $drive_instance)
    {
        $this->CI = &get_instance();
        $this->drive = &$drive_instance;
        $this->root_id = @$this->get('root')->id;
        $HRM_dir = null;
        if ($HRM_dir = $this->CI->options->get_system('drive.hrm.root')) {
            $HRM_dir = @$this->get($HRM_dir);
        }
        if (!$HRM_dir && @$this->search('HRM Data', true, $this->root_id)[0]) {
            $HRM_dir = $this->createDirectory('HRM Data', $this->root_id);
        }
        if (!$HRM_dir) {
            log_message('error', 'Unable to create GDrive HRM Root');
        }
        $this->hrm_root_id = $HRM_dir->id;
        $this->CI->options->set_system('drive.hrm.root', $this->hrm_root_id);
    }
    public function get(string $id): ?HRMFile
    {
        try {
            $file = $this->drive->files->get($id, ['fields' => implode(',', self::_fileParams)]);
            if ($file) {
                return $this->mapFile($file);
            }
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE GET: ', $e);
        }
        return null;
    }
    public function search(string $file_name, $isDirectory = false, $parent = null): array
    {
        $q = sprintf('name contains \'%s\'', urldecode($file_name));
        if ($isDirectory) {
            $q .= ' and mimeType = \'application/vnd.google-apps.folder\'';
        } else {
            $q .= ' and mimeType != \'application/vnd.google-apps.folder\'';
        }
        if (!is_null($parent)) {
            $q .= sprintf(' and \'%s\' in parents', $parent);
        }
        $nextPageToken = null;
        $fileList = [];
        try {
            do {
                $optParams = array(
                    'q' => $q,
                    'pageSize' => 20,
                    'fields' => 'nextPageToken, files(' . implode(',', self::_fileParams) . ')',
                    'pageToken' => $nextPageToken,
                );
                $results = $this->drive->files->listFiles($optParams);
                foreach ($results->getFiles() as $file) {
                    $fileList[] = $this->mapFile($file);
                }
                $nextPageToken = $results->nextPageToken;
            } while (!is_null($nextPageToken));
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE SEARCH: ', $e);
        }
        return $fileList;
    }
    public function getBinary(string $file_id): ?string
    {
        try {
            $res = $this->drive->files->get($file_id, ['alt' => 'media'])->getBody();
            if ($res) {
                $body = '';
                while (!$res->eof()) {
                    $body .= $res->read(1024);
                }
                return $body;
            }
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE GETBINARY: ', $e);
        }
        return null;
    }
    public function getHRMRootId(): ?string
    {
        return $this->hrm_root_id;
    }
    public function all(): array
    {
        $nextPageToken = null;
        $fileList = [];
        try {
            do {
                $optParams = array(
                    'q' => 'trashed=false',
                    'supportsAllDrives' => false,
                    'includeItemsFromAllDrives' => false,
                    'pageSize' => 20,
                    'fields' => 'nextPageToken, files(' . implode(',', self::_fileParams) . ')',
                    'pageToken' => $nextPageToken,
                );
                $results = $this->drive->files->listFiles($optParams);
                foreach ($results->getFiles() as $file) {
                    $fileList[] = $this->mapFile($file);
                }
                $nextPageToken = $results->nextPageToken;
            } while (!is_null($nextPageToken));
            return $fileList;
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE ALL: ', $e);
            return [];
        }
    }
    public function create(string $filepath, $parent = null): ?HRMFile
    {
        $file = new DriveFile();
        $file->name = basename($filepath);
        if (!is_null($parent)) {
            $file->parents = [$parent];
        }
        $file->mimeType = mime_content_type($filepath);
        $data = @file_get_contents($filepath);
        if (!$data) {
            return null;
        }
        try {
            if ($res = $this->drive->files->create($file, ['fields' => 'id', 'data' => $data, 'uploadType' => 'multipart'])) {
                return $this->get($res->id);
            }
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE CREATE: ', $e);
        }
        return null;
    }
    public function rename(string $fileId, string $name): ?HRMFile
    {
        $file = new DriveFile();
        $file->setName($name);
        try {
            if ($res = $this->drive->files->update($fileId, $file, ['fields' => 'id'])) {
                return $this->get($res->id);
            }
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE RENAME: ', $e);
        }
        return null;
    }
    public function createDirectory(string $dirName, $parent = null): ?HRMFile
    {
        $file = new DriveFile();
        $file->name = $dirName;
        if (!is_null($parent)) {
            $file->parents = [$parent];
        }
        $file->mimeType = 'application/vnd.google-apps.folder';
        try {
            if ($res = $this->drive->files->create($file, ['fields' => 'id'])) {
                return $this->get($res->id);
            }
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE CREATE DIR: ', $e);
        }
        return null;
    }
    public function delete(string $file_id): bool
    {
        try {
            $drive_file = new DriveFile();
            $drive_file->trashed = true;
            $this->drive->files->update($file_id, $drive_file);
            return true;
        } catch (\Throwable $e) {
            $this->log('HRM DRIVE DELETE: ', $e);
        }
        return false;
    }
    public function stream(string $file_id): void
    {
        $file = $this->get($file_id);
        if (!$file) {
            exit('not found');
        }
        if (!$file_raw = $this->CI->cache->get('dtd.' . sha1($file_id))) {
            $fileContent = $this->getBinary($file_id);
            if (!$fileContent) {
                exit('file not found');
            }
            $file_raw = base64_encode($fileContent);
            $this->CI->cache->save('dtd.' . sha1($file_id), $file_raw, 300);
        }
        header('Pragma: public'); // required
        header('Expires: 0'); // no cache
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $file->modifiedTime) . ' GMT');
        header('Cache-Control: private', false);
        header('Content-Type: ' . $file->mimeType); // Add the mime type from Code igniter.
        header('Content-Disposition: inline; filename="' . $file->name . '"'); // Add the file name
        header('Content-Transfer-Encoding: binary');
        header('Content-Length: ' . $file->size); // provide file size
        header('Connection: close');
        echo base64_decode($file_raw);
        exit();
    }
    public static function createURL(string $id): string
    {
        return sprintf('https://drive.google.com/open?id=%s', $id);
    }
    private function mapFile($file)
    {
        $HRMFile = new HRMFile();
        $HRMFile->id = $file->id;
        $HRMFile->name = $file->name;
        $HRMFile->size = $file->size ?? 0;
        $HRMFile->mimeType = $file->mimeType;
        $HRMFile->fileExtension = $file->fileExtension ?? '';
        $HRMFile->trashed = $file->trashed ?? false;
        $HRMFile->parents = $file->parents ?? [];
        $HRMFile->createdTime = strtotime($file->createdTime);
        $HRMFile->modifiedTime = strtotime($file->modifiedTime);
        $HRMFile->url = $this->createURL($file->id);
        return $HRMFile;
    }
    private function log($message, ?Throwable $e = null)
    {
        if ($e) {
            $message = $message . ' ' . $e->getMessage() . ' ' . $e->getTraceAsString();
        }
        log_message('error', $message);
    }
}
