<?php
/**
 * Common interface for HRM to talk to devices and perform necessary actions
 */
interface BioDeviceInterface
{
    /**
     * Initialize class and save IP port
     *
     * @param string $ip IP address of device
     * @param int $port Port of device
     * @return null
     */
    public function __construct(string $ip, int $port);
    /**
     * Perform connection to device with saved IP and Port
     *
     * @return bool returns bool to indicate success or error on connection
     */
    public function connect(): bool;
    /**
     * Check if device is connected and reachable
     *
     * @return bool returns true on reachable and false on no connection
     */
    public function isLive(): bool;
    /**
     * Returns device times in UNIX timestamp in seconds
     *
     * @return int
     */
    public function getTime(): int;
    /**
     * set device time
     *
     * @param int $seconds time to set (UNIX timestamp in seconds)
     * @return bool
     */
    public function setTime(int $seconds): bool;
    /**
     * Get user by uid of machine
     *
     * @param int $uid
     * @return ?array user information in the form of associative array or null
     * example:
     * [
     *  'uid' =>
     *  'id' =>
     *  'hrm_id' =>
     *  'name' =>
     *  'role' =>
     * ]
     */
    public function getUser(int $uid): ?array;
    /**
     * Get user by HRM ID
     *
     * @param string $hrm_id
     * @return ?array user information in the form of associative array or null
     * example:
     * [
     *  'uid' =>
     *  'id' =>
     *  'hrm_id' =>
     *  'name' =>
     *  'role' =>
     * ]
     */
    public function getUserByHrmId(string $hrm_id): ?array;

    /**
     * Check if the provided HRM has finger enrolled into machine
     *
     * @param string $hrm_id
     * @return bool return true if any of the finger is enrolled else false
     */
    public function isUserEnrolled(string $hrm_id): bool;
    /**
     * Create user on device.
     * Employee name string on device must be in this format <HRM ID><space><Name>
     *
     * @param int $employee_id Primary key from HRM
     * @param string $hrm_id 8 character employee ID from HRM
     * @param string $name employee name
     * @param string $password pin/password to save on device
     *
     * @return bool
     */
    public function createUser(int $employee_id, string $hrm_id, string $name, string $password, $role = 'user'): bool;
    /**
     * Remove user from machine
     *
     * @param int $employee_id Primary key from HRM
     * @param string 8 character employee ID from HRM
     *
     * @return bool
     */
    public function removeUser(int $employee_id, string $hrm_id = null): bool;
    /**
     * Enable device for use
     *
     * @return bool
     */
    public function enableDevice(): bool;
    /**
     * Disable device for performing system changing tasks or tasks which take time
     *
     * @return bool
     */
    public function disableDevice(): bool;
    /**
     * Return attendance data from device
     *
     * @return ?array multi-dimensional array of associative attendance array
     * Example:
     * [
     *  'id' [int] =>
     *  'uid' [int] =>
     *  'state' [string] => finger,face,password or card
     *  'type' [string] => in or out
     *  'time' [int] => unix timestamp in seconds
     * ]
     */
    public function getAttendance(): ?array;
    public function clearAttendance(): bool;
    public function log(string $message): bool;
}
