<?php
use PHPMailer\PHPMailer\PHPMailer;

class TemplateMailer
{
    private $CI;
    private $mail;
    private $config_arr = [];
    private $logpath = APPPATH . 'logs/mail/';
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->mail = new PHPMailer();
        $this->mail->SMTPDebug = false;
        $this->mail->XMailer = ' ';
        if (!is_dir($this->logpath)) {
            mkdir($this->logpath);
        }
    }
    public function config($config = 0)
    {
        $this->config_arr = $this->CI->db->select('email_templates.*, divisions.name as division_name')->where('division_id', $config)->join('divisions', 'email_templates.division_id=divisions.id', 'left')->get('email_templates')->row_array();
        if ($this->config_arr['isSMTP']) {
            $this->mail->isSMTP(); //Send using SMTP
            $this->mail->SMTPAuth = true; //Enable SMTP authentication
            $this->mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS; //Enable TLS encryption; `PHPMailer::ENCRYPTION_SMTPS` encouraged
        }
        $this->mail->Host = $this->config_arr['host']; //Set the SMTP server to send through
        $this->mail->Username = $this->config_arr['username']; //SMTP username
        $this->mail->Password = $this->CI->encryption->decrypt($this->config_arr['password']); //SMTP password
        $this->mail->Port = $this->config_arr['port']; //TCP port to connect to, use 465 for `PHPMailer::ENCRYPTION_SMTPS` above
        $this->mail->isHTML(true);
        return $this;
    }
    public function get_config($key)
    {
        return $this->config_arr[$key];
    }
    public function send_everyone($division = null, $subject, $body, $attachments = [], $type = 'bcc')
    {
        $type = strtolower($type);
        if ($division) {
            $division = [$division];
            $this->CI->db->where_in('id', $division);
        }
        $divisions = $this->CI->db->select('id')->order_by('id', 'asc')->get('divisions')->result();

        if (!$divisions) {
            return false;
        }

        $divisions = array_column($divisions, 'id');
        $employees = $this->CI->db->select('business_email')
            ->where_in('division_id', $divisions)
            ->where_not_in('status', ['released', 'terminated', 'not-joined', 'resigned'])
            ->get('employee_history')->result();
        if (!$employees) {
            $this->log('No employees found');
            return false;
        }
        $employees = array_column($employees, 'business_email');
        $employees = array_unique($employees);
        $employees = array_filter($employees, function ($e) {
            return (!empty(trim($e)) && filter_var($e, FILTER_VALIDATE_EMAIL));
        });

        $this->config($divisions[0]);
        if (empty(trim($this->config_arr['email_hr']))) {
            $this->log(sprintf('No HR email configured. Failed to send email to hr with subject "%s"', $subject));
            return false;
        }
        $to = [$this->config_arr['email_hr']];
        if (count($divisions) > 1) {
            $division_hrs = $this->CI->db->select('email_hr')->where_in('email_templates.division_id', $divisions)->get('email_templates')->result();
            $division_hrs = array_column($division_hrs, 'email_hr');
            $to = array_merge($to, $division_hrs);
            $to = array_unique($to);
        }
        $employee_chunks = array_chunk($employees, 100 - count($to));
        foreach ($employee_chunks as $employees_c) {
            $this->_send($to, null, $subject, $body, $type == 'cc' ? $employees_c : [], $type == 'bcc' ? $employees_c : [], $attachments);
        }
        return true;
    }
    public function send_to_hr($division, $subject, $body, $cc = [], $bcc = [], $all_division_hr_staff = true, $all_hr_staff = false, $send_bcc = false)
    {
        $this->config($division);
        if (empty(trim($this->config_arr['email_hr']))) {
            $this->log(sprintf('No HR email configured. Failed to send email to hr with subject "%s"', $subject));
            return false;
        }
        if ($all_division_hr_staff) {
            $this->CI->load->model('Employee/Employees_model');

            $hr_emails = $this->CI->Employees_model
                ->get_employee_by_user_id($this->CI->notify->department('hr', !$all_hr_staff ? $division : null)->get_all_users(), 'employee.business_email');
            $this->CI->notify->clear();
            if ($hr_emails) {
                $hr_emails = array_column($hr_emails, 'business_email');
            }

            if ($send_bcc) {
                if (!is_array($bcc)) {
                    $cc = [$bcc];
                }
                $bcc = array_merge($bcc, $hr_emails);
                $bcc = array_unique($bcc);
            } else {
                if (!is_array($cc)) {
                    $cc = [$cc];
                }
                $cc = array_merge($cc, $hr_emails);
                $cc = array_unique($cc);
            }
        }
        return $this->_send($this->config_arr['email_hr'], null, $subject, $body, $cc, $bcc);
    }
    public function send_to_department($division, $department, $subject, $body, $cc = [], $bcc = [], $all_staff = false, $attachments = [], $delete_attachment = false, $to = [])
    {
        $this->config($division);
        if (empty(trim($this->config_arr['email_hr']))) {
            $this->log(sprintf('No HR email configured. Failed to send email to hr with subject "%s"', $subject));
            return false;
        }

        $this->CI->load->model('Employee/Employees_model');
        $all_users = $this->CI->notify->department($department, !$all_staff ? $division : null)->get_all_users();
        if (!$all_users) {
            return false;
        }
        $hr_emails = $this->CI->Employees_model
            ->get_employee_by_user_id($all_users, 'employee.business_email');
        $this->CI->notify->clear();
        if ($hr_emails) {
            $hr_emails = array_column($hr_emails, 'business_email');
            $cc = array_merge($cc, $hr_emails);
        }
        $cc = array_unique($cc);

        return $this->_send($to ? $to : $cc, null, $subject, $body, $to ? $cc: [], $bcc, $attachments, $delete_attachment);
    }
    public function send_to_employee($subject, $body, $employee_id, $direct_supervisor = false, $indirect_supervisor = false, $hr = false, $to_email = 'employee')
    {
        $select_str = 'employee_history.division_id, employee_history.business_email as employee';
        $this->CI->db->where('employee_history.employee_id', $employee_id)->from('employee_history');
        if ($direct_supervisor) {
            $select_str .= ', direct_supervisor.business_email as direct_supervisor';
            $this->CI->db->join('employee_history direct_supervisor', 'employee_history.super_visor_id=direct_supervisor.employee_id', 'left');
        }
        if ($indirect_supervisor) {
            $select_str .= ', indirect_supervisor.business_email as indirect_supervisor';
            $this->CI->db->join('employee_history indirect_supervisor', 'employee_history.indirect_super_visor_id=indirect_supervisor.employee_id', 'left');
        }
        $emp_rec = $this->CI->db->select($select_str)->get()->row_array();
        $this->config($emp_rec['division_id']);
        if ($hr) {
            $emp_rec['hr'] = $this->config_arr['email_hr'];
        }
        $emp_rec = array_filter($emp_rec);
        $cc = [];
        foreach ($emp_rec as $key => $value) {
            if ($key == $to_email) {
                continue;
            }
            if($value == $emp_rec['division_id']) {
                continue;
            }
            $cc[] = $value;
        }
        return $this->_send($emp_rec[$to_email], null, $subject, $body, $cc);
    }
    public function send($job_id, $template_category, $to, $cc = [], $bcc = [], $candidate_id = null, $employee_id = null, $additional_can_param = [])
    {
        $job = $this->CI->db->where('jp_id', $job_id)->get('job_posting')->row();
        $division_id = $job->division_id;
        $this->config($division_id);
        $template = $this->get_template($division_id, $template_category);
        if (!$template) {
            $this->log("Template '$template_category' not found for division id $division_id");
            return false;
        }
        $this->template_category = $template_category;
        $candidate = $this->get_candidate($candidate_id);
        $employee = $employee_id;

        $body = $template->template;
        $body = preg_replace_callback(
            "|{{\S+?}}|",
            function ($matches) use ($candidate, $employee, $additional_can_param) {
                $matches = strtolower(trim(trim(strip_tags($matches[0]), '{}')));
                $matches = explode(':', strtolower($matches));
                if (isset($additional_can_param[$matches[0]]) && isset($additional_can_param[$matches[0]][$matches[1]])) {
                    return $additional_can_param[$matches[0]][$matches[1]];
                } elseif ($matches[0] == 'candidate' && isset($candidate[$matches[1]])) {
                    return $candidate[$matches[1]];
                } elseif ($matches[0] == 'employee' && isset($employee->{$matches[1]})) {
                    return $employee->{$matches[1]};
                }
            },
            $body);
        $template->subjects = preg_replace_callback(
            "|{{\S+?}}|",
            function ($matches) use ($candidate, $employee, $additional_can_param) {
                $matches = strtolower(trim(trim(strip_tags($matches[0]), '{}')));
                $matches = explode(':', strtolower($matches));
                if (isset($additional_can_param[$matches[0]]) && isset($additional_can_param[$matches[0]][$matches[1]])) {
                    return $additional_can_param[$matches[0]][$matches[1]];
                } elseif ($matches[0] == 'candidate' && isset($candidate[$matches[1]])) {
                    return $candidate[$matches[1]];
                } elseif ($matches[0] == 'employee' && isset($employee->{$matches[1]})) {
                    return $employee->{$matches[1]};
                }
            },
            $template->subjects);

        if ($template->cc_hr) {
            $cc[] = $this->config_arr['email_hr'];
            $cc = array_unique($cc);
        }
        return $this->_send($to ? $to : $candidate['email'], null, $template->subjects, $body, $cc, $bcc);
    }

    public function test_email_settings($to, $subject, $body)
    {
        return $this->__send($to, $this->config_arr['email_from'], $subject, $body, $this->config_arr['email_hr']);
    }

    public function _send($to, $from = null, $subject, $body, $cc = [], $bcc = [], $attachments = [], $delete_attachment = false)
    {
        if (!is_array($to)) {
            $to = [$to];
        }
        if (!is_array($cc)) {
            $cc = [$cc];
        }
        if (!is_array($bcc)) {
            $bcc = [$bcc];
        }
        if (!is_array($attachments)) {
            $attachments = [$attachments];
        }
        return $this->CI->db->insert('email_queue', [
            'division_id' => $this->config_arr['division_id'],
            'eto' => implode(',', $to),
            'efrom' => $from ?? $this->config_arr['email_from'],
            'subject' => $subject,
            'content' => json_encode(['cc' => implode(',', $cc), 'bcc' => implode(',', $bcc), 'body' => $body, 'attachments' => json_encode($attachments), 'delete_attachment' => $delete_attachment]),
            'added_at' => date('Y-m-d H:i:s'),
        ]);
    }

    public function __send($to, $from, $subject, $body, $cc = [], $bcc = [], $attachments = [])
    {
        if ($this->template_category) {
            $this->log(sprintf('Sending email for event %s', $this->template_category));
        } else {
            $this->log('Sending email');
        }

        $this->mail->setFrom($from, 'Human Resources');
        if (is_array($cc)) {
            foreach ($cc as $ccc) {
                $this->mail->addCC($ccc);
            }
        } else {
            $this->mail->addCC($cc);
        }

        if (is_array($bcc)) {
            foreach ($bcc as $bccc) {
                $this->mail->addBCC($bccc);
            }
        } else {
            $this->mail->addBCC($bcc);
        }

        if (is_array($to)) {
            foreach ($to as $tat) {
                $this->mail->addAddress($tat);
            }
        } else {
            $this->mail->addAddress($to);
        }

        foreach ($attachments as $key => $attachment) {
            $this->mail->AddAttachment($attachment, is_numeric($key) ? '' : $key);
        }

        $this->mail->Subject = $subject;
        $this->mail->Body = $body;
        try {
            $res = $this->mail->send();
            if ($res) {
                $this->log('Email sent successfully');
                return true;
            } else {
                throw new Exception($this->mail->ErrorInfo);
            }
        } catch (\Throwable $e) {
            $this->log($e->getMessage());
            $this->CI->session->set_userdata('email_exception', $e->getMessage());
            return false;
        } finally {
            $this->mail = new PHPMailer();
        }
    }
    private function log($message)
    {
        @file_put_contents(
            $this->logpath . date('Ymd') . '.log',
            sprintf("%s| %s\n", date('c'), $message),
            FILE_APPEND
        );
    }
    private function get_template($division_id, $temp_cat)
    {
        return $this->CI->db->select('document_template.*, template_category.cc_hr')
            ->join('template_category', 'template_category.id=document_template.category_id', 'left')
            ->where('template_category.keyword', $temp_cat)
            ->where('document_template.division_id', $division_id)
            ->get('document_template')->row();
    }
    private function get_candidate($id)
    {
        $rec = $this->CI->db->select(
            'candidate_basic_info.first_name,
            candidate_basic_info.last_name,
            candidate_basic_info.email,
            candidate_basic_info.phone,
            candidate_basic_info.alter_phone as alternative_phone,
            candidate_basic_info.address,
            candidate_basic_info.total_years_of_experience as experience,
            candidate_basic_info.applied_on,
            position.position_name as position'
        )
            ->join('job_posting', 'job_posting.jp_id=candidate_basic_info.jp_id', 'left')
            ->join('position', 'position.pos_id=job_posting.pos_id', 'left')
            ->where('candidate_basic_info.can_id', $id)
            ->get('candidate_basic_info')->row_array();

        $rec['applied_on'] = formatted_date($rec['applied_on']);

        $rec['interview_date'] = $this->CI->db->select('candidate_shortlist.interview_date')
            ->where('candidate_shortlist.can_id', $id)
            ->where('candidate_shortlist.interview_done', 0)
            ->get('candidate_shortlist')->row_array()['interview_date'];
        $rec['interview_date'] = formatted_date($rec['interview_date'], true);
        return $rec;
    }
}
