<?php

use PhpOffice\PhpSpreadsheet\Style\Alignment;

class Budget_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();

        $this->load->model([
            'payroll/payroll_model' => 'payroll',
        ]);
    }
    public function get($id)
    {
        return $this->db->select('payroll.*, divisions.name as division')
            ->where(['payroll.id' => $id, 'type' => 'budget'])
            ->join('divisions', 'payroll.division_id=divisions.id', 'left')
            ->get('payroll')->row();
    }
    public function queue($division_id, $start, $end, $employees, $departments)
    {
        $employees = array_filter($employees, function ($val) {
            return $val != 'all';
        });
        $departments = array_filter($departments, function ($val) {
            return $val != 'all';
        });
        $this->db->trans_start();
        $this->db->insert('payroll', [
            'type' => 'budget',
            'month' => date('F', strtotime($start)),
            'year' => date('Y', strtotime($start)),
            'start' => sql_date($start),
            'end' => sql_date($end),
            'division_id' => $division_id,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        $id = $this->db->insert_id();
        $this->db->insert('payroll_budget', [
            'payroll_id' => $id,
            'employees' => implode(',', $employees),
            'departments' => implode(',', $departments),
        ]);
        $this->activity
            ->set_division($division_id)
            ->set_payroll($id)
            ->log('budget_added');
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function create($id)
    {
        $payroll = $this->get($id);
        if (!$payroll) {
            show_404();
        }
        $employees = $this->get_records($id);

        $days = date_range($payroll->start, $payroll->end, '+1 day', 'Y-m-d');

        $prepared_data = [];
        $comments_data = [];

        foreach ($employees as $serial => $employee) {
            $prepared_data[] = [
                $serial + 1,
                $employee->hrm_id,
                $employee->name,
                $employee->center,
                formatted_date($employee->hire_date),
                $employee->position,
                $employee->department,
                $employee->project,
                count($days),
                $employee->actual_presents,
                $employee->suspensions,
                $employee->actual_absents,
                $employee->paid_leaves + $employee->special_leaves,
                $employee->gross_salary,
                $employee->basic_salary,
                $employee->actual_salary,
                $employee->arrears + $employee->advances + $employee->incentives,
                $employee->attendance_salary_after_arrears,
                $employee->income_tax,
                $employee->eobi,
                $employee->ncns_penalty,
                $employee->others,
                $employee->provident_fund,
                $employee->deductions,
                $employee->net_salary,
                $employee->account_title,
                $employee->account_number,
                @$employee->account_iban,
                $employee->branch_address,
                $employee->branch_code,
                cnic_with_dashes($employee->cnic),
                $employee->net_salary,
            ];

            $this->db->where_in('column', ['arrears', 'advances', 'incentives']);
            $comments_data[] = [
                16 => getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['payroll_id' => $payroll->id, 'employee_id' => $employee->employee_id], ['updated_at', 'asc']),
                20 => getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['column' => 'ncns', 'payroll_id' => $payroll->id, 'employee_id' => $employee->employee_id], ['updated_at', 'asc']),
                21 => getByWhereAsArray('payroll_adjustments', 'user_name, reason, value', ['column' => 'others', 'payroll_id' => $payroll->id, 'employee_id' => $employee->employee_id], ['updated_at', 'asc']),
            ];
        }

        return [
            'rows' => $prepared_data,
            'comments' => $comments_data,
        ];
    }
    public function get_records($id)
    {
        $employees = $this->payroll->get_payroll_employees($id);
        $inactive_employees = $this->payroll->get_payroll_employees($id, false);
        $employees = array_merge($employees, $inactive_employees);

        return array_map(function ($employee) {
            $employee->deductions = 0;
            $employee->attendance_salary_after_arrears = $employee->actual_salary + $employee->arrears + $employee->advances + $employee->incentives;

            $employee->deductions += $employee->income_tax;
            $employee->deductions += $employee->eobi;
            $employee->provident_fund = $employee->provident_fund;
            $employee->deductions += $employee->provident_fund;
            $employee->deductions += $employee->ncns_penalty;
            $employee->deductions += round($employee->others);

            $employee->net_salary = $employee->attendance_salary_after_arrears - $employee->deductions;
            return $employee;
        }, $employees);
    }
    public function download($id)
    {
        $this->load->library('better_excel');

        $payroll = $this->get($id);
        if (!$payroll) {
            show_404();
        }

        $data = $this->create($id);

        $this->better_excel->download(
            $this->get_header(),
            $data['rows'],
            $this->get_meta(),
            [],
            $data['comments'],
            sprintf('Budget_Payroll_%s_to_%s', formatted_date($payroll->start), formatted_date($payroll->end)),
            'D2',
            [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            86
        );
    }
    public function get_meta()
    {
        require_once APPPATH . 'libraries/Mysql_type.php';
        $mysql_types = new Mysql_type();
        return [
            $mysql_types::TYPE_INT24, /* 'S.No' */
            $mysql_types::TYPE_VARCHAR, /* 'Empl. ID' */
            $mysql_types::TYPE_VARCHAR, /* 'Name' */
            $mysql_types::TYPE_VARCHAR, /* 'Center' */
            $mysql_types::TYPE_DATE, /* 'Date of Joining' */
            $mysql_types::TYPE_VARCHAR, /* 'Designation' */
            $mysql_types::TYPE_VARCHAR, /* 'Department' */
            $mysql_types::TYPE_VARCHAR, /* 'Project' */
            $mysql_types::TYPE_INT24, /* 'Total Days' */
            $mysql_types::TYPE_DECIMAL, /* 'Total No of Days Present' */
            $mysql_types::TYPE_INT24, /* 'Total No Days Suspensions' */
            $mysql_types::TYPE_DECIMAL, /* 'Total No Days Absent/Unpaid Leave' */
            $mysql_types::TYPE_DECIMAL, /* 'Paid Leaves' */
            $mysql_types::TYPE_INT24, /* 'Gross Salary' */
            $mysql_types::TYPE_INT24, /* 'Basic Salary' */
            $mysql_types::TYPE_INT24, /* 'Actual Salary as per Attendance', */
            $mysql_types::TYPE_INT24, /* 'Arrears (if any)' */
            $mysql_types::TYPE_INT24, /* 'Budgeted Salary', */
            $mysql_types::TYPE_INT24, /* 'Income Tax', */
            $mysql_types::TYPE_INT24, /* 'EOBI', */
            $mysql_types::TYPE_INT24, /* 'Penalty on No Call No Show', */
            $mysql_types::TYPE_INT24, /* 'Other Deductions (If any)', */
            $mysql_types::TYPE_INT24, /* 'Provident Fund', */
            $mysql_types::TYPE_INT24, /* 'Total Deduction', */
            $mysql_types::TYPE_INT24, /* 'Net Salary', */
            $mysql_types::TYPE_VARCHAR, /* 'Title of Account', */
            $mysql_types::TYPE_VARCHAR, /* 'Account Number', */
            $mysql_types::TYPE_VARCHAR, /* 'Branch Address', */
            $mysql_types::TYPE_VARCHAR, /* 'Branch Code', */
            $mysql_types::TYPE_VARCHAR, /* 'CNIC #', */
            $mysql_types::TYPE_INT24, /* 'Net Salary' */
        ];
    }
    public function get_header()
    {
        return [
            ['S.No', 5],
            ['Empl. ID', 10],
            ['Name', 25],
            ['Center', 15],
            'Date of Joining',
            'Designation',
            'Department',
            'Project',
            'Total Days',
            'Total No of Days Present',
            'Total No Days Suspensions',
            'Total No Days Absent/Unpaid Leave',
            'Paid Leaves',
            'Gross Salary',
            'Basic Salary',
            'Actual Salary as per Attendance',
            'Arrears (if any)',
            'Budgeted Salary',
            'Income Tax',
            ['EOBI', 6],
            'Penalty on No Call No Show',
            'Other Deductions (If any)',
            'Provident Fund',
            'Total Deduction',
            'Net Salary',
            'Title of Account',
            'Account Number',
            ['Branch Address', 30],
            ['Branch Code', 15],
            ['CNIC #', 16],
            'Net Salary',
        ];
    }
    public function generate_budget($payroll_id, $cron = false)
    {
        $this->db->trans_start();
        $payroll = $this->db
            ->select('payroll.*, payroll_budget.employees, payroll_budget.departments')
            ->where([
                'payroll.id' => $payroll_id,
                'type' => 'budget',
            ])
            ->join('payroll_budget', 'payroll.id=payroll_budget.payroll_id', 'left')
            ->get('payroll')->row();
        if (!$payroll) {
            return false;
        }
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_attendance');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_employee');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_used_leaves');
        $this->db->where(['payroll_id' => $payroll_id, 'type' => 'system'])->delete('payroll_adjustments');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_payorders');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_payslips');

        $months = interval_into_months($payroll->start, $payroll->end);
        $years = interval_into_years($payroll->start, $payroll->end);

        $employees_arr = [];
        foreach ($months as $month) {
            /**
             * Active Employees
             */
            $employees = $this->payroll->get_active_employees($payroll->division_id, $month[0], $month[count($month) - 1], $cron, $payroll->employees, $payroll->departments);
            foreach ($employees as $employee) {
                $per_days_salary = round(($employee->current_salary ?? 0) / cal_days_in_month(CAL_GREGORIAN, date('m', strtotime($month[0])), date('Y', strtotime($month[0]))), 2);
                $gross_salary = round($per_days_salary * count($month));
                $basic_salary = percentage_to_amount($gross_salary ?? 0, 68);

                $attendance = $this->generate_budget_attendance($employee, $payroll_id, $month[0], $month[count($month) - 1], true);
                $paid_days = $this->payroll->get_paid_days((object) $attendance);

                $actual_salary = round($paid_days * $per_days_salary);

                if (!isset($employees_arr[$employee->employee_id])) {
                    $employees_arr[$employee->employee_id] = [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'hrm_id' => $employee->hrm_id,
                        'name' => sprintf('%s %s', $employee->first_name, $employee->last_name),
                        'cnic' => $employee->cnic ?? '',
                        'center' => $employee->city ?? '',
                        'dob' => $employee->dob ?? '',
                        'hire_date' => $employee->hire_date,
                        'duty_type' => $employee->duty_type,
                        'position' => $employee->position ?? '',
                        'department' => $employee->department ?? '',
                        'project' => $employee->primary_project ?? '',
                        'leave_quota' => $this->payroll->get_leave_quota($employee->employee_id, $month[0]) * count($years),
                        'actual_gross_salary' => $_actual_gross = calculate_gross_salary($employee->employee_id, $month[0], null, $cron),
                        'actual_basic_salary' => percentage_to_amount($_actual_gross, 68),
                        'gross_salary' => $gross_salary,
                        'basic_salary' => $basic_salary,
                        'actual_salary' => $actual_salary,
                        'income_tax' => calculate_income_tax($employee->current_salary),
                        'eobi' => ($employee->eobi_check == 1 ? calculate_eobi($employee->eobi ?? 0, $employee->hire_date, $month[count($month) - 1], $employee->training_period) : 0),
                        'provident_fund' => ($employee->duty_type != 5 && $employee->provident_fund_check == 1) ? round(percentage_to_amount($basic_salary ?? 0, 9.3)) : 0,
                        'account_title' => $employee->account_title ?? '',
                        'account_number' => $employee->account_number ?? '',
                        'account_iban' => @$employee->account_iban ?? '',
                        'branch_address' => $employee->branch_address ?? '',
                        'branch_code' => $employee->branch_code ?? '',
                        'presents' => $attendance['presents'],
                        'paid_leaves' => $attendance['paid_leaves'],
                        'special_leaves' => $attendance['special_leaves'],
                        'offs' => $attendance['offs'],
                        'ncns' => $attendance['ncns'],
                        'suspensions' => $attendance['suspensions'],
                        'late_arrivals' => $attendance['late_arrivals'],
                        'unpaid_leaves' => $attendance['unpaid_leaves'],
                        'half_days' => $attendance['half_days'],
                        'unpaid_half_days' => $attendance['unpaid_half_days'],
                        'under_training' => $attendance['under_training'],
                    ];
                } else {
                    $employees_arr[$employee->employee_id]['gross_salary'] += $gross_salary;
                    $employees_arr[$employee->employee_id]['basic_salary'] += $basic_salary;
                    $employees_arr[$employee->employee_id]['actual_salary'] += $actual_salary;
                    $employees_arr[$employee->employee_id]['income_tax'] += calculate_income_tax($employee->current_salary);
                    $employees_arr[$employee->employee_id]['eobi'] += ($employee->eobi_check == 1 ? calculate_eobi($employee->eobi ?? 0, $employee->hire_date, $month[count($month) - 1], $employee->training_period) : 0);
                    $employees_arr[$employee->employee_id]['provident_fund'] += $employee->duty_type == 1 ? round(percentage_to_amount($basic_salary ?? 0, 9.3)) : 0;

                    $employees_arr[$employee->employee_id]['presents'] += $attendance['presents'];
                    $employees_arr[$employee->employee_id]['paid_leaves'] += $attendance['paid_leaves'];
                    $employees_arr[$employee->employee_id]['special_leaves'] += $attendance['special_leaves'];
                    $employees_arr[$employee->employee_id]['offs'] += $attendance['offs'];
                    $employees_arr[$employee->employee_id]['ncns'] += $attendance['ncns'];
                    $employees_arr[$employee->employee_id]['suspensions'] += $attendance['suspensions'];
                    $employees_arr[$employee->employee_id]['late_arrivals'] += $attendance['late_arrivals'];
                    $employees_arr[$employee->employee_id]['unpaid_leaves'] += $attendance['unpaid_leaves'];
                    $employees_arr[$employee->employee_id]['half_days'] += $attendance['half_days'];
                    $employees_arr[$employee->employee_id]['unpaid_half_days'] += $attendance['unpaid_half_days'];
                    $employees_arr[$employee->employee_id]['under_training'] = $attendance['under_training'];
                }
            }

            /**
             * Non-active employees
             */
            $employees = $this->payroll->get_nonactive_employees($payroll->division_id, $month[0], $month[count($month) - 1], $cron, $payroll->employees, $payroll->departments);
            foreach ($employees as $employee) {
                $per_days_salary = round($employee->current_salary / cal_days_in_month(CAL_GREGORIAN, date('m', strtotime($month[0])), date('Y', strtotime($month[0]))), 2);
                $gross_salary = round($per_days_salary * count($month));
                $basic_salary = percentage_to_amount($gross_salary ?? 0, 68);

                $attendance = $this->generate_budget_attendance($employee, $payroll_id, $month[0], $month[count($month) - 1], true);
                $paid_days = $this->payroll->get_paid_days((object) $attendance);

                $actual_salary = round($paid_days * $per_days_salary);

                if (!isset($employees_arr[$employee->employee_id])) {
                    $employees_arr[$employee->employee_id] = [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'status' => $employee->status == 'terminated' ? 'terminated' : 'resigned',
                        'hrm_id' => $employee->hrm_id,
                        'name' => sprintf('%s %s', $employee->first_name, $employee->last_name),
                        'cnic' => $employee->cnic ?? '',
                        'center' => $employee->city ?? '',
                        'dob' => $employee->dob ?? '',
                        'hire_date' => $employee->hire_date,
                        'duty_type' => $employee->duty_type,
                        'position' => $employee->position ?? '',
                        'department' => $employee->department ?? '',
                        'project' => $employee->primary_project ?? '',
                        'leave_quota' => $this->payroll->get_leave_quota($employee->employee_id, $month[0]) * count($years),
                        'actual_gross_salary' => $_actual_gross = calculate_gross_salary($employee->employee_id, $month[0], null, $cron),
                        'actual_basic_salary' => percentage_to_amount($_actual_gross, 68),
                        'gross_salary' => $gross_salary,
                        'basic_salary' => $basic_salary,
                        'actual_salary' => $actual_salary,
                        'income_tax' => calculate_income_tax($gross_salary),
                        'eobi' => ($employee->eobi_check == 1 ? calculate_eobi($employee->eobi ?? 0, highest_date($employee->hire_date, $month[0]), $employee->termination_date, $employee->training_period) : 0),
                        'provident_fund' => 0,
                        'account_title' => $employee->account_title ?? '',
                        'account_number' => $employee->account_number ?? '',
                        'account_iban' => @$employee->account_iban ?? '',
                        'branch_address' => $employee->branch_address ?? '',
                        'branch_code' => $employee->branch_code ?? '',
                        'presents' => $attendance['presents'],
                        'paid_leaves' => $attendance['paid_leaves'],
                        'special_leaves' => $attendance['special_leaves'],
                        'offs' => $attendance['offs'],
                        'ncns' => $attendance['ncns'],
                        'suspensions' => $attendance['suspensions'],
                        'late_arrivals' => $attendance['late_arrivals'],
                        'unpaid_leaves' => $attendance['unpaid_leaves'],
                        'half_days' => $attendance['half_days'],
                        'unpaid_half_days' => $attendance['unpaid_half_days'],
                        'under_training' => $attendance['under_training'],
                    ];
                } else {
                    $employees_arr[$employee->employee_id]['gross_salary'] += $gross_salary;
                    $employees_arr[$employee->employee_id]['basic_salary'] += $basic_salary;
                    $employees_arr[$employee->employee_id]['actual_salary'] += $actual_salary;
                    $employees_arr[$employee->employee_id]['income_tax'] += calculate_income_tax($gross_salary);
                    $employees_arr[$employee->employee_id]['eobi'] += ($employee->eobi_check == 1 ? calculate_eobi($employee->eobi ?? 0, highest_date($employee->hire_date, $month[0]), $employee->termination_date, $employee->training_period) : 0);

                    $employees_arr[$employee->employee_id]['presents'] += $attendance['presents'];
                    $employees_arr[$employee->employee_id]['paid_leaves'] += $attendance['paid_leaves'];
                    $employees_arr[$employee->employee_id]['special_leaves'] += $attendance['special_leaves'];
                    $employees_arr[$employee->employee_id]['offs'] += $attendance['offs'];
                    $employees_arr[$employee->employee_id]['ncns'] += $attendance['ncns'];
                    $employees_arr[$employee->employee_id]['suspensions'] += $attendance['suspensions'];
                    $employees_arr[$employee->employee_id]['late_arrivals'] += $attendance['late_arrivals'];
                    $employees_arr[$employee->employee_id]['unpaid_leaves'] += $attendance['unpaid_leaves'];
                    $employees_arr[$employee->employee_id]['half_days'] += $attendance['half_days'];
                    $employees_arr[$employee->employee_id]['unpaid_half_days'] += $attendance['unpaid_half_days'];
                    $employees_arr[$employee->employee_id]['under_training'] = $attendance['under_training'];
                }
            }
        }
        foreach ($employees_arr as $employee) {
            $this->db->insert('payroll_employee', encrypt_employee_data($employee, [], 'payroll_employee'));
            foreach ($years as $year) {
                $this->payroll->generate_used_leaves($payroll_id, $employee['employee_id'], $year);
                $this->payroll->generate_used_special_leaves($payroll_id, $employee['employee_id'], $year);
            }
        }
        $this->db->where('id', $payroll_id)->update('payroll', [
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function generate_budget_attendance($employee, $payroll_id, $start, $end)
    {
        $days_range = date_range($start, $end, '+1 day', 'Y-m-d');
        $weekends = $this->payroll->get_weekends();
        $holidays = $this->payroll->get_holidays($start, $end);
        $attendance = $this->payroll->get_attendance($employee->employee_id, $start, $end);
        $suspensions = $this->payroll->get_suspensions($employee->employee_id, $start, $end);
        $leaves = $this->payroll->get_leaves($employee->employee_id, $start, $end);
        $special_leaves = $this->payroll->get_special_leaves($employee->employee_id, $start, $end);
        $penalties = $this->payroll->get_penalties($employee->employee_id, $start, $end);

        if ($penalties) {
            foreach ($penalties as $penalty) {
                if ($penalty->amount == 0) {
                    continue;
                }
                if ($penalty->amount > 0) {
                    $this->db->where(['payroll_id' => $payroll_id, 'employee_id' => $employee->employee_id])->update('payroll_employee', [
                        'incentives_changed' => 1,
                    ]);
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'incentives',
                        'value' => $penalty->amount,
                        'type' => 'system',
                        'reason' => $penalty->title . ' - ' . $penalty->genID,
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                } else {
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'others',
                        'value' => abs($penalty->amount),
                        'type' => 'system',
                        'reason' => $penalty->title . ' - ' . $penalty->genID,
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);

                }
            }
        }

        $count_presents = 0;
        $count_paid_leaves = 0;
        $count_special_leaves = 0;
        $count_unpaid_leaves = 0;
        $count_suspensions = 0;
        $count_ncns = 0;
        $count_late = 0;
        $count_offs = 0;
        $count_half_days = 0;
        $count_unpaid_half_days = 0;

        $hire_date = new DateTime($employee->hire_date);
        $actual_tenure = $tenure = $hire_date->diff(new DateTime(end($days_range)))->format('%a');
        reset($days_range);

        $start_of_today = strtotime('00:00:00');
        $absent_15_tenure_penalty_deducted = false;
        foreach ($days_range as $day) {
            if (strtotime($day) < strtotime($employee->hire_date)) {
                continue;
            }
            if ($employee->termination_date && strtotime($day) >= strtotime($employee->termination_date)) {
                $this->db->insert('payroll_attendance', [
                    'payroll_id' => $payroll_id,
                    'employee_id' => $employee->employee_id,
                    'day' => $day,
                    'flag' => $this->payroll->get_absent_flag_by_status($employee->status, $day, $employee->termination_date),
                ]);
                continue;
            }

            if ($tenure <= ($employee->training_period + 60)) {
                $tenure = $hire_date->diff(new DateTime($day))->format('%a');
            }

            $flag = null;
            if (isset($suspensions[$day])) {
                $flag = 'sus';
                $count_suspensions++;
            } elseif (isset($holidays[$day]) && in_array($employee->shift, $holidays[$day])) {
                $flag = 'off';
                $count_offs++;
            } elseif (in_array(date('l', strtotime($day)), $weekends[$employee->shift] ?? $weekends[0])) {
                $flag = 'off';
                $count_offs++;
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'full_day') {
                $flag = 'pl';
                $count_special_leaves++;
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'half_day') {
                if (isset($attendance[$day]) && (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day]) || in_array('UH-D', $attendance[$day]))) {
                    $flag = 'h-d';
                    $count_half_days += 0.5;
                    $count_special_leaves += 0.5;
                } else {
                    $flag = 'hpl';
                    $count_unpaid_leaves += 0.5;
                    $count_unpaid_half_days += 0.5;
                }
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'full_day') {
                $flag = 'pl';
                $count_paid_leaves++;
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'half_day') {
                if (isset($attendance[$day]) && (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day]) || in_array('UH-D', $attendance[$day]))) {
                    $flag = 'h-d';
                    $count_half_days += 0.5;
                    $count_paid_leaves += 0.5;
                } else {
                    $flag = 'hpl';
                    $count_unpaid_leaves += 0.5;
                    $count_unpaid_half_days += 0.5;
                }
            } elseif (isset($attendance[$day])) {
                if (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day])) {
                    $count_presents++;
                    $flag = 'p';
                } elseif (in_array('UH-D', $attendance[$day])) {
                    $count_unpaid_leaves += 0.5;
                    $count_unpaid_half_days += 0.5;
                    $flag = 'uh-d';
                } elseif (in_array('A', $attendance[$day])) {
                    if (strtotime($day) >= $start_of_today) {
                        $flag = 'p';
                        $count_presents++;
                    } else {
                        $flag = 'a';
                        $count_ncns++;
                    }
                } elseif (in_array('P', $attendance[$day])) {
                    $count_presents++;
                    $flag = 'p';
                } elseif (in_array('LA', $attendance[$day])) {
                    $count_late++;
                    $flag = 'la';
                } else {
                    if (strtotime($day) >= $start_of_today) {
                        $flag = 'p';
                        $count_presents++;
                    } else {
                        $flag = 'a';
                        $count_ncns++;
                    }
                }
                if ($flag == 'p' && in_array('LA', $attendance[$day])) {
                    $count_presents--;
                    $count_late++;
                    $flag = 'la';
                }
            } else {
                if (strtotime($day) >= $start_of_today) {
                    $flag = 'p';
                    $count_presents++;
                } else {
                    $flag = 'a';
                    $count_ncns++;
                }
            }
            if ($employee->training_period > 0 && $tenure == ($employee->training_period + 1) && !$absent_15_tenure_penalty_deducted) {
                $last_training_day = date('Y-m-d', strtotime($day . ' -1 day'));

                $fifteen_days_attendance = $this->get_attendance_flag_in_period(
                    $employee,
                    $start,
                    $end
                );
                if (in_array('a', $fifteen_days_attendance)) {
                    $day_salary = $employee->current_salary / count($days_range);
                    $day_15_salary = round($day_salary * getWeekdayDifference($employee->shift, $employee->hire_date, $last_training_day));
                    $this->db->insert('payroll_adjustments', [
                        'payroll_id' => $payroll_id,
                        'employee_id' => $employee->employee_id,
                        'column' => 'others',
                        'value' => $day_15_salary,
                        'type' => 'system',
                        'reason' => sprintf('Auto deducted %s days (%s - %s) of salary for absence within %s days of joining date.', $employee->training_period, formatted_date($employee->hire_date), formatted_date($last_training_day), $employee->training_period),
                        'user_id' => 0,
                        'user_name' => 'HRM PAYROLL SYSTEM',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                    $absent_15_tenure_penalty_deducted = true;
                }
            }
            $this->db->insert('payroll_attendance', [
                'payroll_id' => $payroll_id,
                'employee_id' => $employee->employee_id,
                'day' => $day,
                'flag' => $flag,
            ]);
        }
        return [
            'presents' => $count_presents,
            'paid_leaves' => $count_paid_leaves,
            'special_leaves' => $count_special_leaves,
            'offs' => $count_offs,
            'ncns' => $count_ncns,
            'suspensions' => $count_suspensions,
            'late_arrivals' => $count_late,
            'unpaid_leaves' => $count_unpaid_leaves,
            'half_days' => $count_half_days,
            'unpaid_half_days' => $count_unpaid_half_days,
            'under_training' => ($employee->training_period > 0 && $actual_tenure < $employee->training_period) ? 1 : 0,
        ];
    }
    public function get_attendance_flag_in_period($employee, $start, $end)
    {
        $days_range = date_range($start, $end, '+1 day', 'Y-m-d');
        $weekends = $this->payroll->get_weekends();
        $holidays = $this->payroll->get_holidays($start, $end);
        $attendance = $this->payroll->get_attendance($employee->employee_id, $start, $end);
        $suspensions = $this->payroll->get_suspensions($employee->employee_id, $start, $end);
        $leaves = $this->payroll->get_leaves($employee->employee_id, $start, $end);
        $special_leaves = $this->payroll->get_special_leaves($employee->employee_id, $start, $end);

        $flags = [];
        reset($days_range);
        $start_of_today = strtotime('00:00:00');
        foreach ($days_range as $day) {
            if (strtotime($day) < strtotime($employee->hire_date)) {
                continue;
            }
            if ($employee->termination_date && strtotime($day) >= strtotime($employee->termination_date)) {
                continue;
            }

            $flag = null;
            if (isset($suspensions[$day])) {
                $flag = 'sus';
            } elseif (isset($holidays[$day]) && in_array($employee->shift, $holidays[$day])) {
                $flag = 'off';
            } elseif (in_array(date('l', strtotime($day)), $weekends[$employee->shift] ?? $weekends[0])) {
                $flag = 'off';
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'full_day') {
                $flag = 'pl';
            } elseif (isset($special_leaves[$day]) && $special_leaves[$day] == 'half_day') {
                $flag = 'h-d';
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'full_day') {
                $flag = 'pl';
            } elseif (isset($leaves[$day]) && $leaves[$day] == 'half_day') {
                $flag = 'h-d';
            } elseif (isset($attendance[$day])) {
                if (in_array('FD', $attendance[$day]) || in_array('OT', $attendance[$day])) {
                    $flag = 'p';
                } elseif (in_array('UH-D', $attendance[$day])) {
                    $flag = 'uh-d';
                } elseif (in_array('A', $attendance[$day])) {
                    if (strtotime($day) >= $start_of_today) {
                        $flag = 'p';
                    } else {
                        $flag = 'a';
                    }
                } elseif (in_array('P', $attendance[$day])) {
                    $flag = 'p';
                } elseif (in_array('LA', $attendance[$day])) {
                    $flag = 'la';
                } else {
                    if (strtotime($day) >= $start_of_today) {
                        $flag = 'p';
                    } else {
                        $flag = 'a';
                    }
                }
                if ($flag == 'p' && in_array('LA', $attendance[$day])) {
                    $flag = 'la';
                }
            } else {
                if (strtotime($day) >= $start_of_today) {
                    $flag = 'p';
                } else {
                    $flag = 'a';
                }
            }
            $flags[] = $flag;
        }
        return $flags;
    }
    public function delete($payroll_id)
    {
        $this->db->trans_start();
        $payroll = $this->db->select('division_id, start, end')->where('id', $payroll_id)->get('payroll')->row();
        $this->db->where('id', $payroll_id)->delete('payroll');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_attendance');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_employee');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_used_leaves');
        $this->db->where(['payroll_id' => $payroll_id, 'type' => 'system'])->delete('payroll_adjustments');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_payorders');
        $this->db->where('payroll_id', $payroll_id)->delete('payroll_payslips');

        $this->activity
            ->set_division($payroll->division_id)
            ->log([
                'budget_deleted',
                $payroll_id,
                $payroll->start,
                $payroll->end,
            ]);
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
}
