<?php
class Evaluation_report_model extends CI_Model
{
    const _TYPES = [
        'semiannually' => 12,
        'biannually' => 6,
        'triannually' => 4,
        'quarterly' => 3,
    ];
    const __WORDING = [
        'biannually' => [
            1 => 'Mid Year',
            2 => 'Yearly',
        ],
    ];
    const _SELECTED_TYPE = 'biannually';

    public function auto_generate()
    {
        $employees = $this->get_employees();
        $this->db->trans_start();
        foreach ($employees as $employee) {
            $this->generate($employee->employee_id);
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function generate($employee_id, $custom_year = null)
    {
        $employee = $this->get_employee($employee_id);
    
        if($custom_year) {
            $employee->tenure_in_months = ($custom_year - date('Y', strtotime($employee->hire_date))) * 12;
            $employee->tenure_in_months_after_confirmation = $employee->tenure_in_months - $employee->probation_period;
            $employee->tenure_cycle = ceil($employee->tenure_in_months_after_confirmation / 12);
            $employee->report_cycle = 12;
        }
        
        $periods = $this->get_report_periods();
        $nums = $this->get_current_report_counts($employee->report_cycle);
        
        if (!$nums) {
            return;
        }
        $current_num = end($nums);
        reset($nums);

        $year = date($custom_year ? $custom_year : 'Y');
        if ($current_num > 1) {
            $year = date($custom_year ? $custom_year : 'Y', strtotime('-' . ($current_num * $periods[1][2]) . ' months'));
        }
        $report_parent = $this->get_year_report($employee->employee_id, $employee->tenure_cycle, $year);

        foreach ($nums as $num) {
            $months_interval = $this->get_period_interval($employee->confirmation_date, $year, $periods[$num]);
            $report_instance_exists = $this->db
                ->where([
                    'ev_id' => $report_parent->id,
                    'employee_id' => $employee->employee_id,
                    'num' => $num,
                ])
                ->count_all_results('evaluation_report_instance');
            if ($report_instance_exists > 0) {
                continue;
            }
            $deadline = getWeekdayPeriod($employee->shift_id, 5)['end'];
            $report_employee = $this->db->where(['report_id' => $report_parent->id, 'employee_id' => $employee->employee_id])->get('report_employee')->row();
            if (!$report_employee) {
                $this->db->insert('report_employee', [
                    'report_id' => $report_parent->id,
                    'employee_id' => $employee->employee_id,
                    'dept_id' => $employee->dept_id ?? 0,
                    'department' => $employee->department ?? '',
                    'pos_id' => $employee->pos_id ?? 0,
                    'position' => $employee->position ?? '',
                ]);
            } else {
                $this->db->where(['report_id' => $report_parent->id, 'employee_id' => $employee->employee_id])->update('report_employee', [
                    'dept_id' => $employee->dept_id,
                    'department' => $employee->department,
                    'pos_id' => $employee->pos_id,
                    'position' => $employee->position,
                ]);
            }
            $this->db->insert('evaluation_report_instance', [
                'ev_id' => $report_parent->id,
                'employee_id' => $employee->employee_id,
                'num' => $num,
                'start' => $months_interval['start'],
                'end' => $months_interval['end'],
                'type' => $this::__WORDING[$this::_SELECTED_TYPE][$num],
                'status' => 'pending-employee',
                'dm_id' => $employee->super_visor_id,
                'idm_id' => $employee->indirect_super_visor_id,
                'added_at' => date('Y-m-d H:i:s'),
                'next_deadline' => $deadline . ' 23:59:59',
            ]);
            $instance_id = $this->db->insert_id();
            if(!$this->options->get_system('evaluation.test.mode')) {
                $this->notify
                    ->employee($employee->employee_id)
                    ->send(
                        [
                            'evaluation_generated_employee',
                            $deadline,
                        ],
                        'employee/evaluation/pending/' . $instance_id
                    );
                if ($this->options->get_system('evaluation.notifications.email') && $employee->business_email) {
                    $this->template_mail
                        ->send_to_employee(
                            'Evaluation Report Generated',
                            $this->load->view('employee/email/report_generated_employee', ['employee' => (array) $employee, 'deadline' => $deadline], true),
                            $employee->employee_id,
                            true,
                        );
                }
            }
        }
    }
    public function refresh_report($id)
    {
        $res = $this->db->query(
            'CALL refresh_report_data(?);',
            [$id]
        );
        $this->db->next_result();
        $system_opt_slug = sprintf('eval_obj_data_fetch_%s_', $id);
        $this->db->like('option_key', $system_opt_slug, 'after')->delete('system_options');
        return $res;
    }
    public function get_year_report($employee_id, $tenure_cycle, $year)
    {
        $year_rec = $this->db->where([
            'employee_id' => $employee_id,
            'tenure_year' => $tenure_cycle,
        ])->get('evaluation_report')->row();
        if (!$year_rec) {
            $res = $this->db->insert('evaluation_report', [
                'employee_id' => $employee_id,
                'year' => $year,
                'tenure_year' => $tenure_cycle,
            ]);
            if (!$res) {
                return null;
            }
            $this->db->query(
                'CALL refresh_report_data(?);',
                [$this->db->insert_id()]
            );
            return $this->get_year_report($employee_id, $tenure_cycle, $year);
        }
        return $year_rec;
    }
    public function get_current_report_counts($current_month)
    {
        $counts = [];

        for ($i = 1; $i <= $current_month; $i++) {
            $count = floor($i / $this::_TYPES[$this::_SELECTED_TYPE]);
            if ($count > 0 && !in_array($count, $counts)) {
                $counts[] = $count;
            }
        }
        return $counts;
    }
    public function get_report_periods()
    {
        $counts = [];

        for ($i = 1; $i <= 12; $i++) {
            $count = floor(($i - 1) / $this::_TYPES[$this::_SELECTED_TYPE]) + 1;
            if (!isset($counts[$count])) {
                $counts[$count] = [];
            }
            $counts[$count][] = $i;
        }
        foreach ($counts as &$count) {
            $len = count($count);
            $count = array_values(array_filter($count, function ($k) use ($len) {
                return ($k == 0) || ($k == ($len - 1));
            }, ARRAY_FILTER_USE_KEY));
        }
        return $counts;
    }
    public function get_employees()
    {
        return $this->db
            ->select(
                'employee_id' .
                (isset($_GET['debug']) ? ', tenure_in_months,
                IF(tenure_in_months - probation_period - 1 < 1, 0, IF(((tenure_in_months - probation_period - 1) % 12) < 1, 12, (tenure_in_months - probation_period - 1) % 12)) as report_cycle,
                CEIL(GREATEST((tenure_in_months - probation_period) - 1, 0) / 12) as tenure_cycle' : null)
            )
            ->where([
                'status' => 'active',
                'duty_type_id' => 1,
            ])
            ->where('tenure_in_months - probation_period > ', 6, false)
            ->get('employee_details')->result();
    }
    public function get_employee($employee_id)
    {
        return $this->db
            ->select([
                'employee_id',
                'full_name',
                'shift_id',
                'department',
                'division_id',
                'division',
                'position',
                'business_email',
                'pos_id',
                'dept_id',
                'super_visor_id',
                'indirect_super_visor_id',
                'hire_date',
                'probation_period',
                'DATE_ADD(hire_date, INTERVAL probation_period MONTH) as confirmation_date',
                '(tenure_in_months - 1) as tenure_in_months',
                '(tenure_in_months - probation_period - 1) as tenure_in_months_after_confirmation',
                'IF(tenure_in_months - probation_period - 1 < 1, 0, IF(((tenure_in_months - probation_period - 1) % 12) < 1, 12, (tenure_in_months - probation_period - 1) % 12)) as report_cycle',
                'CEIL((tenure_in_months - probation_period - 1) / 12) as tenure_cycle',
            ])
            ->where([
                'employee_id' => $employee_id,
            ])
            ->get('employee_details')->row();
    }
    public function email_deadlines()
    {
        $expired_deadlines = $this->db->select(
            'evaluation_report_instance.id,
            evaluation_report_instance.employee_id,
            evaluation_report_instance.status,
            evaluation_report_instance.next_deadline,
            evaluation_report_instance.dm_id,
            evaluation_report_instance.dm_date,
            evaluation_report_instance.idm_id,
            evaluation_report_instance.idm_date,
            CONCAT_WS(\' \', employee_history.first_name, employee_history.last_name) as full_name,
            employee_history.hrm_id,
            employee_history.business_email,
            employee_history.division_id')
            ->join('employee_history', 'evaluation_report_instance.employee_id=employee_history.employee_id', 'left')
            ->where(['evaluation_report_instance.next_deadline <' => date('Y-m-d H:i:s'), 'evaluation_report_instance.deadline_notified' => 0])
            ->get('evaluation_report_instance')
            ->result();

        $this->db->trans_start();
        foreach ($expired_deadlines as $report) {
            $data = [];
            $data['deadline'] = formatted_date($report->next_deadline);
            $data['id'] = $report->id;
            $data['employee'] = null;
            switch ($report->status) {
                case 'pending-employee':{
                        $data['actor'] = [
                            'full_name' => $report->full_name,
                            'hrm_id' => $report->hrm_id,
                            'division_id' => $report->division_id,
                            'business_email' => $report->business_email,
                        ];
                        break;
                    }
                case 'pending-dm':{
                        $data['actor'] = $this->db->select('full_name, hrm_id, business_email, division_id')->where('employee_id', $report->dm_id)->get('employee_details')->row_array();
                        $data['employee'] = [
                            'full_name' => $report->full_name,
                            'hrm_id' => $report->hrm_id,
                        ];
                        break;
                    }
                case 'pending-idm':{
                        $data['actor'] = $this->db->select('full_name, hrm_id, business_email, division_id')->where('employee_id', $report->idm_id)->get('employee_details')->row_array();
                        $data['employee'] = [
                            'full_name' => $report->full_name,
                            'hrm_id' => $report->hrm_id,
                        ];
                        break;
                    }
                default:{
                        continue 2;
                    }
            }
            $data['url'] = rtrim(get_division_domain($data['actor']['division_id']), '\/\\') . '/employee/evaluation/pending/' . $report->id;
            if(!$this->options->get_system('evaluation.test.mode')) {               
                $this->notify
                    ->department('hr')
                    ->send(
                        [
                            'evaluation_expired_hr',
                            $data['actor']['full_name'],
                            $data['actor']['hrm_id'],
                        ],
                        'employee/evaluation/pending/' . $report->id
                    );
                if ($this->options->get_system('evaluation.notifications.email')) {
                    $this->template_mail->send_to_hr(
                        $report->division_id,
                        sprintf('%s: %s (%s)', display(['Evaluation', 'Report', 'Review', 'Deadline', 'Expired']), $data['actor']['full_name'], $data['actor']['hrm_id']),
                        $this->load->view('employee/email/report_expired_hr', $data, true),
                        $data['actor']['business_email'],
                        [],
                        true,
                        true,
                        true
                    );
                }
            }
            $this->db->where('id', $report->id)->update('evaluation_report_instance', [
                'deadline_notified' => 1,
            ]);
        }
        $this->db->trans_complete();
        return $this->db->trans_status();
    }
    public function get_period_interval($hire_date, $year, $period)
    {
        $data = [];

        $a = $period[0] - 1;
        $b = $period[1] - 1;
        if ($a < 1) {
            $data['start'] = $year . date('-m-d', strtotime($hire_date));
        } else {
            $data['start'] = $year . date('-m-d', strtotime($hire_date . ' +' . $a . ' month'));
        }
        $data['end'] = $year . date('-m-d', strtotime($hire_date . ' +' . $b . ' month'));

        if(strtotime($data['end']) < strtotime($data['start'])) {
            $data['end'] = date('Y', strtotime($data['start'] . ' + 1 year')) . date('-m-d', strtotime($data['end']));
        }
        return $data;
    }
}
