<?php

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class Bulk extends Loggedin_Controller
{
    private $allowed_flags = [
        'P',
        'A',
        'LA',
        'UH-D',
        'H-D',
        'SPL',
        'OFF'
    ];
    public function __construct()
    {
        parent::__construct();
        $this->load->model([
            'attendence_model' => 'attendance_model',
        ]);
    }
    public function import()
    {
        $this->permission->method('bulk_attendance_import', 'create')->redirect();
        if ($_FILES['file']) {
            $this->_import();return;
        }
        $month_first_last = month_first_last_day();
        $data['columns'] = date_range($month_first_last['first'], $month_first_last['last'], '+1 day', 'd-m-y');
        $data['allowed_flags'] = $this->allowed_flags;
        $data['employees'] = array_column(get_random_employee(5), 'hrm_id');

        $data['title'] = display('bulk_attendance_import');
        $data['months'] = month_labels();
        $data['years'] = range(date('Y', strtotime('+1 years')), date('Y', strtotime('-30 years')));
        $data['module'] = 'attendance';
        $data['page'] = "bulk/import";
        echo Modules::run('template/layout', $data);
    }
    public function sample()
    {
        $this->permission->method('bulk_attendance_import', 'create')->redirect();

        $date = null;
        if ($this->input->post('month') && $this->input->post('year')) {
            $date = date('Y-m-d', mktime(0, 0, 0, $this->input->post('month'), 1, $this->input->post('year')));
        } else {
            $date = date('Y-m-d');
        }

        $spreadsheet = new Spreadsheet();
        $properties = $spreadsheet->getProperties();
        $properties->setCreator('HRM System');
        $properties->setLastModifiedBy('HRM System');
        $properties->setTitle('Attendance Bulk Import');
        $properties->setSubject('Attendance Bulk Import');
        $properties->setDescription('Auto generated sample sheet for HRM Attendance Bulk Import System');

        $sheet = $spreadsheet->getActiveSheet();

        $sheet->setTitle(date('F, Y', strtotime($date)));
        $sheet->freezePane('B2');
        $sheet->getSheetView()->setZoomScale(90);
        $sheet->getRowDimension('1')->setRowHeight(40);

        $month_first_last = month_first_last_day(date('m', strtotime($date)), date('Y', strtotime($date)));
        $columns = date_range($month_first_last['first'], $month_first_last['last']);

        array_unshift($columns, 'Employee ID');

        $last_column = null;
        foreach ($columns as $column => $value) {
            $cell = $sheet->getCellByColumnAndRow($column + 1, 1);
            $sheet->getColumnDimension($cell->getColumn())->setWidth(15);
            $last_column = $cell->getColumn();
            switch ($column) {
                case 0:{
                        $cell->setValue($value);
                        break;
                    }
                default:{
                        $timestamp = \PhpOffice\PhpSpreadsheet\Shared\Date::PHPToExcel($value);
                        if (!$timestamp) {
                            $cell->setValueExplicit(
                                $value,
                                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                            );
                        } else {
                            $cell->setValue($timestamp);
                            $cell->getStyle()->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_DATE_SHORT);
                        }
                        break;
                    }
            }
        }
        if ($last_column) {
            $sheet->getStyle("A1:{$last_column}1")->applyFromArray([
                'fill' => [
                    'color' => [
                        'rgb' => 'C5BE97',
                    ],
                    'fillType' => Fill::FILL_SOLID,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => [
                            'rgb' => '000000',
                        ],
                    ],
                ],
                'font' => [
                    'bold' => 1,
                    'size' => 12,
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                    'wrapText' => 1,
                ],
            ]);
        }
        $sheet->getStyle("A2");
        $writer = new Xlsx($spreadsheet);
        header('Content-Description: File Transfer');
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="Attendance Import - ' . date('F Y', strtotime($date)) . '.xlsx"');
        header('Content-Transfer-Encoding: binary');
        $writer->save('php://output');
    }
    private function _import()
    {
        $this->load->library('employee/myupload');
        $file = $this->myupload->do_upload(FCPATH . 'uploads/temp/', 'file', true);
        if (!$file) {
            $this->session->set_flashdata('exception', 'File upload failed, try again later!');
            redirect($_SERVER['HTTP_REFERER']);
        }
        $reader = IOFactory::createReader('Xlsx');
        $reader->getReadDataOnly(true);
        $spreadsheet = $reader->load($file);
        $sheet = $spreadsheet->getSheet(0);

        $header = [];
        $data = [];
        $shifts = $this->get_shifts();
        foreach ($sheet->getRowIterator() as $row) {
            $employee = null;
            foreach ($row->getCellIterator() as $cell) {
                if ($cell->getRow() == 1) {
                    //Header
                    switch ($cell->getCoordinate()) {
                        case 'A1':{
                                $label = str_slug($cell->getValue());
                                if ($label != 'employee-id') {
                                    $this->session->set_flashdata('exception', 'First Column of first row (A1) must be Employee ID in AAA-1111 format');
                                    redirect($_SERVER['HTTP_REFERER']);
                                }
                                $header[$cell->getColumn()] = $label;
                                break;
                            }
                        default:{
                                $value = trim($cell->getValue());
                                if(empty($value)) {
                                    break;
                                }
                                $timestamp = Date::excelToTimestamp($value);
                                if (!$timestamp) {
                                    $this->session->set_flashdata('exception', 'Invalid value in header cell (' . $cell->getCoordinate() . '). Cell must contain a valid date format');
                                    redirect($_SERVER['HTTP_REFERER']);
                                }
                                $header[$cell->getColumn()] = date('Y-m-d', Date::excelToTimestamp($value));
                                break;
                            }
                    }
                } else {
                    switch ($cell->getColumn()) {
                        case 'A':{
                                if (empty(trim($cell->getValue()))) {
                                    continue 3;
                                }
                                $employee = $this->db->select('employee_id, hrm_id, shift, super_visor_id as direct_supervisor')->where('hrm_id', $cell->getValue())->get('employee_history')->row();
                                if (!$employee) {
                                    $this->session->set_flashdata('exception', sprintf('No employee with ID \'%s\' found. Please verify Employee ID on %s', $cell->getValue(), $cell->getCoordinate()));
                                    redirect($_SERVER['HTTP_REFERER']);
                                }
                                if (!$employee->shift || !$shifts[$employee->shift]) {
                                    $this->session->set_flashdata('exception', sprintf('Invalid shift is assigned to employee with ID \'%s\' on %s. Please verify.', $employee->hrm_id, $cell->getCoordinate()));
                                    redirect($_SERVER['HTTP_REFERER']);
                                }
                                $data['e' . $employee->employee_id] = [
                                    'hrm_id' => $employee->hrm_id,
                                    'shift' => $employee->shift,
                                    'employee_id' => $employee->employee_id,
                                    'direct_supervisor' => $employee->direct_supervisor,
                                    'attendance' => [],
                                ];
                                break;
                            }
                        default:{
                                if (!$header[$cell->getColumn()]) {
                                    continue 2;
                                }
                                $value = strtoupper(trim($cell->getValue()));
                                if (empty($value)) {
                                    continue 2;
                                }
                                if (!in_array($value, $this->allowed_flags)) {
                                    $this->session->set_flashdata('exception', sprintf('Invalid attendance type \'%s\' found in cell %s. Only %s are allowed.', $cell->getValue(), $cell->getCoordinate(), implode(', ', $this->allowed_flags)));
                                    redirect($_SERVER['HTTP_REFERER']);
                                }
                                $data['e' . $employee->employee_id]['attendance'][] = [
                                    'date' => $header[$cell->getColumn()],
                                    'flag' => $value,
                                ];
                                break;
                            }
                    }
                }
            }
        }
        $this->db->trans_start();
        foreach ($data as $employee) {
            $shift = $shifts[$employee['shift']];
            if ($shift->flexible_shift) {
                $shift_start = '00:00:00';
                $shift_end = date('H:i:s', strtotime('00:00:00 +' . $shift->flexible_hours . ' hours'));
            } else {
                $shift_start = $shift->start;
                $shift_end = $shift->end;
            }
            $current_shift_start = null;
            $current_shift_end = null;
            $leave_type_id = $this->get_leave_type_id();
            foreach ($employee['attendance'] as $day) {
                $current_shift_start = $day['date'] . ' ' . $shift_start;
                $current_shift_end = $day['date'] . ' ' . $shift_end;
                $flag_in = 'P';
                $flag_out = 'FD';

                if ($this->attendance_model->is_nightsift($shift)) {
                    $current_shift_end = date('Y-m-d H:i:s', strtotime($current_shift_end . ' +1 day'));
                }
                switch ($day['flag']) {
                    case 'P':{
                            break;
                        }
                    case 'LA':{
                            $flag_in = 'LA';
                            $current_shift_start = date('Y-m-d H:i:s', strtotime($current_shift_start . ' +16 minutes'));
                            break;
                        }
                    case 'UH-D':{
                            $flag_out = 'UH-D';
                            $total_seconds = strtotime($current_shift_end) - strtotime($current_shift_start);
                            $total_seconds = $total_seconds * 0.66;
                            $current_shift_end = date('Y-m-d H:i:s', strtotime($current_shift_start) + $total_seconds);
                            break;
                        }
                    case 'H-D':{
                            if (!$this->leave_exists_for_day($employee['employee_id'], $day['date'])) {
                                $this->db->insert('leave_apply', [
                                    'employee_id' => $employee['employee_id'],
                                    'leave_type_id' => $leave_type_id,
                                    'apply_strt_date' => $day['date'],
                                    'apply_end_date' => $day['date'],
                                    'apply_day' => 1,
                                    'leave_aprv_strt_date' => $day['date'],
                                    'leave_aprv_end_date' => $day['date'],
                                    'num_aprv_day' => 0.5,
                                    'reason' => 'Bulk attendance import ' . formatted_date_now('', true),
                                    'apply_date' => $day['date'],
                                    'approve_date' => formatted_date_now(),
                                    'approved_by' => $this->session->userdata('employee_id') ?? 0,
                                    'supervisor_id' => $employee['direct_supervisor'],
                                    'supervisor_status' => 'pending',
                                    'hr_status' => 'approved',
                                ]);
                                $this->db->insert('leave_applied_days', [
                                    'leave_appl_id' => $this->db->insert_id(),
                                    'day' => $day['date'],
                                    'type' => 'half_day',
                                    'applied' => 1,
                                    'active' => 1,
                                ]);
                            }
                        }
                    case 'SPL':{
                            if (!$this->leave_exists_for_day($employee['employee_id'], $day['date'])) {
                                $this->db->insert('leave_apply', [
                                    'employee_id' => $employee['employee_id'],
                                    'leave_type_id' => $leave_type_id,
                                    'apply_strt_date' => $day['date'],
                                    'apply_end_date' => $day['date'],
                                    'apply_day' => 1,
                                    'leave_aprv_strt_date' => $day['date'],
                                    'leave_aprv_end_date' => $day['date'],
                                    'num_aprv_day' => 1,
                                    'reason' => 'Bulk attendance import ' . formatted_date_now('', true),
                                    'apply_date' => $day['date'],
                                    'approve_date' => formatted_date_now(),
                                    'approved_by' => $this->session->userdata('employee_id') ?? 0,
                                    'supervisor_id' => $employee['direct_supervisor'],
                                    'supervisor_status' => 'pending',
                                    'hr_status' => 'approved',
                                ]);
                                $this->db->insert('leave_applied_days', [
                                    'leave_appl_id' => $this->db->insert_id(),
                                    'day' => $day['date'],
                                    'type' => 'full_day',
                                    'applied' => 1,
                                    'active' => 1,
                                ]);
                            }
                        }
                    default:{
                            continue 2;
                        }
                }
                $this->db->where(['active' => 1, 'uid' => $employee['employee_id'], 'day' => $day['date']])->update('attendance_history', [
                    'active' => 0,
                ]);
                $this->db->query("CALL add_attendance(?, ?, ?, ?, ?, ?);",
                    [
                        $employee['employee_id'],
                        $current_shift_start,
                        $current_shift_end,
                        $flag_in,
                        $flag_out,
                        $this->session->userdata('employee_id') ?? 0,
                    ]
                );
                $this->db->next_result();
                $this->activity->set_employee($employee['employee_id'])->set_url(sprintf('attendance/home/admin_attendance/%s/%s', $employee['employee_id'], date('d-m-Y', strtotime($day['date']))))->log([
                    'full_attendance_inserted',
                    formatted_date($current_shift_start, true),
                    formatted_date($current_shift_end, true),
                ]);
            }
        }
        $this->db->trans_complete();
        if ($this->db->trans_status()) {
            $this->session->set_flashdata('success', 'Attendance imported successfully.');
        } else {
            $this->session->set_flashdata('exception', 'Unable to import attendance, try again!');
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
    private function get_shifts()
    {
        $recs = $this->db->select('id, start, end, flexible_shift, flexible_hours')->get('shift')->result();
        $data = [];
        foreach ($recs as $rec) {
            $data[$rec->id] = $rec;
        }

        return $data;
    }
    private function get_leave_type_id()
    {
        $rec = $this->db->select('leave_type_id')->like('leave_type', 'special')->get('leave_type')->row();
        if (!$rec) {
            $rec = $this->db->select_max('leave_type_id')->get('leave_type')->row();
        }
        if (!$rec) {
            return 1;
        }
        return $rec->leave_type_id;
    }
    private function leave_exists_for_day($employee_id, $day)
    {
        return $this->db->where(['employee_id' => $employee_id, 'leave_aprv_strt_date <= ' => $day, 'leave_aprv_end_date >= ' => $day])->count_all_results('leave_apply') > 0;
    }
}
