<?php
require_once APPPATH . 'libraries/biomatric_devices/BioDeviceInterface.php';

class MyZK implements BioDeviceInterface
{
    private $CI = null;
    private $zk = null;
    private $logpath = APPPATH . 'logs/bio_devices/';
    private $_ip = '';
    private $_port = null;
    private $_password = null;
    private $_users = [];
    private $connected = false;
    private $_cache_string = 'device.';

    public function __construct(string $ip, int $port, string $password = null)
    {
        $this->CI = &get_instance();
        $this->_ip = $ip;
        $this->_port = $port;
        $this->_password = $password;
        $this->_cache_string .= preg_replace('#[^\w\d]+#', '', $ip . $port);
    }
    public function connect(): bool
    {
        if ($this->connected) {
            return true;
        }
        try {
            $this->zk = new ZKLibrary($this->_ip, $this->_port, $this->_password);
            if (!$this->zk->connect()) {
                throw new Exception("Unable to connect to device: {$this->_ip}:{$this->_port}");
            }
            return $this->connected = true;
        } catch (\Throwable $e) {
            $this->log($e->getMessage());
            $this->zk = null;
            $this->connected = false;
            return false;
        }
    }
    public function isLive(): bool
    {
        return $this->_isUp();
    }
    public function getTime(): int
    {
        return strtotime($this->zk->getTime());
    }
    public function setTime(int $seconds): bool
    {
        return $this->zk->setTime(date('Y-m-d H:i:s', $seconds)) !== false;
    }
    public function getUser(int $id): ?array
    {
        if (!$this->_users) {
            $this->_fetchUsers();
        }

        $recs = array_filter($this->_users, function ($us) use (&$id) {return $us['id'] == $id;});
        return $recs ? array_pop($recs) : null;
    }
    public function getUserByHrmId(string $hrm_id): ?array
    {
        if (!$this->_users) {
            $this->_fetchUsers();
        }

        $recs = array_filter($this->_users, function ($us) use (&$hrm_id) {return $us['hrm_id'] == $hrm_id;});
        return $recs ? array_pop($recs) : null;
    }
    public function isUserEnrolled(string $hrm_id): bool
    {
        $cache_string = $this->_cache_string . '.enrolled.' . preg_replace('#\W+#', '', $hrm_id);
        if (!$data = $this->CI->cache->get($cache_string)) {
            $data = [
                'enrolled' => false,
            ];
            $user = $this->getUserByHrmId($hrm_id);

            $finger_order = [5, 6, 7, 8, 9, 4, 3, 2, 1, 0];
            foreach ($finger_order as $finger) {
                $rec = $this->zk->getUserTemplate($user['uid'], $finger);
                if ($rec) {
                    $data['enrolled'] = true;
                    break;
                }
            }
            $this->CI->cache->save($cache_string, json_encode($data), 100);
        } else {
            $data = json_decode($data, true);
        }
        return $data['enrolled'];
    }
    public function getUsers()
    {
        if (!$this->_users) {
            $this->_fetchUsers();
        }

        return array_values($this->_users);
    }
    public function createUser(int $employee_id, string $hrm_id, string $name, string $password, $role = 'user'): bool
    {
        if ($this->getUserByHrmId($hrm_id)) {
            return false;
        }
        if ($this->getUser($employee_id)) {
            return $this->zk->setUser(null, $employee_id, "{$hrm_id} {$name}", $password, $role == 'user' ? $this->zk::LEVEL_USER : $this->zk::LEVEL_SUPERMANAGER) != false;
        } else {
            return $this->zk->setUser($employee_id, $employee_id, "{$hrm_id} {$name}", $password, $role == 'user' ? $this->zk::LEVEL_USER : $this->zk::LEVEL_SUPERMANAGER) != false;
        }
    }
    public function removeUser(int $employee_id, string $hrm_id = null): bool
    {
        if ($hrm_id && $user = $this->getUserByHrmId($hrm_id)) {
            $this->zk->deleteUser($user['uid']);
        }
        return true;
    }
    public function getAttendance(): ?array
    {
        if (!$this->zk) {
            return null;
        }
        $data = $this->zk->getAttendance();
        $attendance = [];

        foreach ($data as $dt) {
            $attendance[] = [
                'id' => $dt[0],
                'uid' => $dt[1],
                'state' => $this->parseState($dt[2]),
                'type' => $this->parseType($dt[4]),
                'time' => strtotime($dt[3]),
            ];
        }
        return $attendance;
    }
    public function enableDevice(): bool
    {
        if (!$this->zk) {
            return false;
        }
        return $this->zk->enableDevice() != false;
    }
    public function disableDevice(): bool
    {
        if (!$this->zk) {
            return false;
        }
        return $this->zk->disableDevice() != false;
    }
    private function _fetchUsers()
    {
        $_users = $this->zk->getUser();
        if (!$_users) {
            $this->log('Users not loaded');return;
        }
        if (!$processed_users = $this->CI->cache->get($this->_cache_string . '.users')) {
            $hrm_ids = [];
            $processed_users = [];
            foreach ($_users as $uid => $user) {
                $processed_users[] = [
                    'uid' => $uid,
                    'id' => $user[0],
                    'hrm_id' => $hrm_id = trim(substr($user[1], 0, strpos($user[1], ' '))),
                    'name' => trim(substr($user[1], strpos($user[1], ' '))),
                    'role' => $user[2],
                ];
                if (!empty(trim($hrm_id))) {
                    $hrm_ids[] = $hrm_id;
                }
            }

            //check for duplicates
            $id_counts = array_count_values($hrm_ids);
            foreach ($id_counts as $key => $value) {
                if ($value > 1) {
                    $this->log("Duplicate HRM ID: {$key} / time: {$value}");
                }
            }
            $this->CI->cache->save($this->_cache_string . '.users', $this->CI->encryption->encrypt(json_encode($processed_users)), 100);
            return $this->_users = $processed_users;
        }
        return $this->_users = json_decode($this->CI->encryption->decrypt($processed_users), true);
    }
    private function _isUp($max_tries = 10)
    {
        if (!$this->zk) {
            return false;
        }
        $res = 'down';
        $try = 0;
        do {
            $res = $this->zk->ping();
            $try++;
        } while ($res == 'down' && $try < $max_tries);

        return $res != 'down';
    }
    private function parseState($state)
    {
        switch ($state) {
            case $this->zk::ATT_STATE_FINGERPRINT:{
                    return 'finger';
                }
            case $this->zk::ATT_STATE_PASSWORD:{
                    return 'password';
                }
            case $this->zk::ATT_STATE_CARD:{
                    return 'card';
                }
            case $this->zk::ATT_STATE_FACE:{
                    return 'face';
                }
            default:
                return null;
        }
    }
    private function parseType($type)
    {
        switch ($type) {
            case $this->zk::ATT_TYPE_CHECK_IN:
            case $this->zk::ATT_TYPE_OVERTIME_IN:{
                    return 'in';
                }
            case $this->zk::ATT_TYPE_CHECK_OUT:
            case $this->zk::ATT_TYPE_OVERTIME_OUT:{
                    return 'out';
                }
            default:
                return null;
        }
    }
    public function log(string $message): bool
    {
        return @file_put_contents(
            $this->logpath . date('Ymd') . '.log',
            sprintf("%s| %s\n", date('c'), $message),
            FILE_APPEND
        ) !== false;
    }
    public function __destruct()
    {
        if (!$this->zk) {
            return false;
        }
        $this->zk->disconnect();
    }
}

class ZKLibrary
{
    const USHRT_MAX = 65535;
    const LEVEL_USER = 0;
    const LEVEL_ENROLLER = 2;
    const LEVEL_MANAGER = 12;
    const LEVEL_SUPERMANAGER = 14;

    const CMD_DB_RRQ = 7; //Read saved data.
    const CMD_USER_WRQ = 8; //Upload user data.
    const CMD_USERTEMP_RRQ = 9; //Read user fingerprint template.
    const CMD_USERTEMP_WRQ = 10; //Upload user fingerprint template.
    const CMD_OPTIONS_RRQ = 11; //Read configuration value of the machine.
    const CMD_OPTIONS_WRQ = 12; //Change configuration value of the machine.
    const CMD_ATTLOG_RRQ = 13; //Request attendance log.
    const CMD_CLEAR_DATA = 14; //Delete data.
    const CMD_CLEAR_ATTLOG = 15; //Delete attendance record.
    const CMD_DELETE_USER = 18; //Delete user.
    const CMD_DELETE_USERTEMP = 19; //Delete user fingerprint template.
    const CMD_CLEAR_ADMIN = 20; //Clears admins privileges.
    const CMD_USERGRP_RRQ = 21; //Read user group.
    const CMD_USERGRP_WRQ = 22; //Set user group.
    const CMD_USERTZ_RRQ = 23; //Get user timezones.
    const CMD_USERTZ_WRQ = 24; //Set the user timezones.
    const CMD_GRPTZ_RRQ = 25; //Get group timezone.
    const CMD_GRPTZ_WRQ = 26; //Set group timezone.
    const CMD_TZ_RRQ = 27; //Get device timezones.
    const CMD_TZ_WRQ = 28; //Set device timezones.
    const CMD_ULG_RRQ = 29; //Get group combination to unlock.
    const CMD_ULG_WRQ = 30; //Set group combination to unlock.
    const CMD_UNLOCK = 31; //Unlock door for a specified amount of time.
    const CMD_CLEAR_ACC = 32; //Restore access control to default.
    const CMD_CLEAR_OPLOG = 33; //Delete operations log.
    const CMD_OPLOG_RRQ = 34; //Read operations log.
    const CMD_GET_FREE_SIZES = 50; //Request machine status (remaining space).
    const CMD_ENABLE_CLOCK = 57; //Enables the ":" in screen clock.
    const CMD_STARTVERIFY = 60; //Set the machine to authentication state.
    const CMD_STARTENROLL = 61; //Start enroll procedure.
    const CMD_CANCELCAPTURE = 62; //Disable normal authentication of users.
    const CMD_STATE_RRQ = 64; //Query state.
    const CMD_WRITE_LCD = 66; //Prints chars to the device screen.
    const CMD_CLEAR_LCD = 67; //Clear screen captions.
    const CMD_GET_PINWIDTH = 69; //Request max size for users id.
    const CMD_SMS_WRQ = 70; //Upload short message.
    const CMD_SMS_RRQ = 71; //Download short message.
    const CMD_DELETE_SMS = 72; //Delete short message.
    const CMD_UDATA_WRQ = 73; //Set user short message.
    const CMD_DELETE_UDATA = 74; //Delete user short message.
    const CMD_DOORSTATE_RRQ = 75; //Get door state.
    const CMD_WRITE_MIFARE = 76; //Write data to Mifare card.
    const CMD_EMPTY_MIFARE = 78; //Clear Mifare card.
    const CMD_VERIFY_WRQ = 79; //Change verification style of a given user.
    const CMD_VERIFY_RRQ = 80; //Read verification style of a given user.
    const CMD_TMP_WRITE = 87; //Transfer fp template from buffer.
    const CMD_CHECKSUM_BUFFER = 119; //Get checksum of machine's buffer.
    const CMD_DEL_FPTMP = 134; //Deletes fingerprint template.
    const CMD_GET_TIME = 201; //Request machine time.
    const CMD_SET_TIME = 202; //Set machine time.
    const CMD_REG_EVENT = 500; //Realtime events.
    const CMD_CONNECT = 1000; //Begin connection.
    const CMD_EXIT = 1001; //Disconnect.
    const CMD_ENABLEDEVICE = 1002; //Change machine state to "normal work".
    const CMD_DISABLEDEVICE = 1003; //Disables fingerprint, rfid reader and keyboard.
    const CMD_RESTART = 1004; //Restart machine.
    const CMD_POWEROFF = 1005; //Shut-down machine.
    const CMD_SLEEP = 1006; //Change machine state to "idle".
    const CMD_RESUME = 1007; //Change machine state to "awaken".
    const CMD_CAPTUREFINGER = 1009; //Capture fingerprint picture.
    const CMD_TEST_TEMP = 1011; //Test if fingerprint exists.
    const CMD_CAPTUREIMAGE = 1012; //Capture the entire image.
    const CMD_REFRESHDATA = 1013; //Refresh the machine stored data.
    const CMD_REFRESHOPTION = 1014; //Refresh the configuration parameters.
    const CMD_TESTVOICE = 1017; //Test voice.
    const CMD_GET_VERSION = 1100; //Request the firmware edition.
    const CMD_CHANGE_SPEED = 1101; //Change transmission speed.
    const CMD_AUTH = 1102; //Request to begin session using commkey.
    const CMD_PREPARE_DATA = 1500; //Prepare for data transmission.
    const CMD_DATA = 1501; //Data packet.
    const CMD_FREE_DATA = 1502; //Release buffer used for data transmission.
    const CMD_DATA_WRRQ = 1503; //Read/Write a large data set.
    const CMD_DATA_RDY = 1504; //Indicates that it is ready to receive data.
    const CMD_ACK_OK = 2000; //The request was processed sucessfully.
    const CMD_ACK_ERROR = 2001; //There was an error when processing the request.
    const CMD_ACK_DATA = 2002;
    const CMD_ACK_RETRY = 2003;
    const CMD_ACK_REPEAT = 2004;
    const CMD_ACK_UNAUTH = 2005; //Connection not authorized.
    const CMD_ACK_ERROR_DATA = 65531;
    const CMD_ACK_ERROR_INIT = 65532;
    const CMD_ACK_ERROR_CMD = 65533;
    const CMD_ACK_UNKNOWN = 65535; //Received unknown command.

    const ATT_STATE_FINGERPRINT = 1;
    const ATT_STATE_PASSWORD = 0;
    const ATT_STATE_CARD = 2;
    const ATT_STATE_FACE = 16;

    const ATT_TYPE_CHECK_IN = 0;
    const ATT_TYPE_CHECK_OUT = 1;
    const ATT_TYPE_OVERTIME_IN = 4;
    const ATT_TYPE_OVERTIME_OUT = 5;

    public $ip = null;
    public $port = null;
    public $password = 0;
    public $socket = null;
    public $session_id = 0;
    public $received_data = '';
    public $user_data = array();
    public $attendance_data = array();
    public $timeout_sec = 20;
    public $timeout_usec = 20000000;

    public function __construct($ip = null, $port = null, $password = null)
    {
        if ($ip != null) {
            $this->ip = $ip;
        }
        if ($port != null) {
            $this->port = $port;
        }
        if ($password != null) {
            $this->password = $password;
        }
        $this->socket = socket_create(AF_INET, SOCK_DGRAM, SOL_UDP);
        if (!$this->socket) {
            $errorcode = socket_last_error();
            $errormsg = socket_strerror($errorcode);
            throw new Exception("Couldn't create socket: [$errorcode] $errormsg");
        }
        $this->setTimeout($this->timeout_sec, $this->timeout_usec);
    }

    public function __destruct()
    {
        unset($this->received_data);
        unset($this->user_data);
        unset($this->attendance_data);
    }

    public function connect($ip = null, $port = null)
    {
        if ($ip != null) {
            $this->ip = $ip;
        }
        if ($port != null) {
            $this->port = $port;
        }
        if ($this->ip == null || $this->port == null) {
            return false;
        }
        $res = $this->sendCommand(self::CMD_CONNECT, '');
        if ($res['status']) {
            if ($res['code'] == self::CMD_ACK_UNAUTH) {
                $command_string = $this->makeCommkey($this->password, $this->session_id);
                $res = $this->sendCommand(self::CMD_AUTH, $command_string);
                if ($res['status'] && $res['code'] == self::CMD_ACK_OK) {
                    return true;
                }
            }
        }
        return false;
    }
    private function makeCommkey($key, $session_id, $ticks = 50)
    {
        $key = intval($key);
        $session_id = intval($session_id);

        $k = 0;
        for ($i = 0; $i < 32; $i++) {
            if ($key & (1 << $i)) {
                $k = ($k << 1 | 1);
            } else {
                $k = $k << 1;
            }
        }

        $k += $session_id;
        $k = pack('I', $k);
        $k = unpack("C*", $k);

        $k = pack(
            "c*",
            $k[1] ^ ord('Z'),
            $k[2] ^ ord('K'),
            $k[3] ^ ord('S'),
            $k[4] ^ ord('O')
        );

        $k = unpack("v*", $k);
        $k = pack("v*", $k[2], $k[1]);

        $B = 0xff & $ticks;
        $k = unpack("c*", $k);
        $k = pack(
            "c*",
            $k[1] ^ $B,
            $k[2] ^ $B,
            $B,
            $k[4] ^ $B);

        return $k;
    }
    public function disconnect()
    {
        if ($this->ip == null || $this->port == null) {
            return false;
        }
        $command = self::CMD_EXIT;
        $command_string = '';
        $chksum = 0;
        $session_id = $this->session_id;
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $reply_id = hexdec($u['h8'] . $u['h7']);
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            return $this->checkValid($this->received_data);
        } catch (\Throwable $e) {
            return false;
        }
    }

    public function sendCommand($command, $command_string)
    {
        $chksum = 0;
        $session_id = $this->session_id;
        $reply_id = -1 + self::USHRT_MAX;
        if ($this->received_data) {
            $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
            $reply_id = hexdec($u['h8'] . $u['h7']);
        }
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            if (strlen($this->received_data) > 0) {
                $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6', substr($this->received_data, 0, 8));
                $this->session_id = hexdec($u['h6'] . $u['h5']);
                $command = hexdec($u['h2'] . $u['h1']);
                return [
                    'status' => true,
                    'code' => $command,
                ];
            } else {
                return [
                    'status' => false,
                    'code' => null,
                ];
            }
        } catch (\Throwable $e) {
            return [
                'status' => false,
                'code' => null,
            ];
        }
    }
    public function setTimeout($sec = 0, $usec = 0)
    {
        if ($sec != 0) {
            $this->timeout_sec = $sec;
        }
        if ($usec != 0) {
            $this->timeout_usec = $usec;
        }
        $timeout = array('sec' => $this->timeout_sec, 'usec' => $this->timeout_usec / 1000000);
        socket_set_option($this->socket, SOL_SOCKET, SO_RCVTIMEO, $timeout);
    }

    public function ping($timeout = 10)
    {
        $time1 = microtime(true);
        $pfile = fsockopen($this->ip, $this->port, $errno, $errstr, $timeout);
        if (!$pfile) {
            return 'down';
        }
        $time2 = microtime(true);
        fclose($pfile);
        return round((($time2 - $time1) * 1000), 0);
    }

    private function reverseHex($input)
    {
        $output = '';
        for ($i = strlen($input); $i >= 0; $i--) {
            $output .= substr($input, $i, 2);
            $i--;
        }
        return $output;
    }

    private function encodeTime($time)
    {
        $str = str_replace(array(":", " "), array("-", "-"), $time);
        $arr = explode("-", $str);
        $year = @$arr[0] * 1;
        $month = ltrim(@$arr[1], '0') * 1;
        $day = ltrim(@$arr[2], '0') * 1;
        $hour = ltrim(@$arr[3], '0') * 1;
        $minute = ltrim(@$arr[4], '0') * 1;
        $second = ltrim(@$arr[5], '0') * 1;
        $data = (($year % 100) * 12 * 31 + (($month - 1) * 31) + $day - 1) * (24 * 60 * 60) + ($hour * 60 + $minute) * 60 + $second;
        return $data;
    }

    private function decodeTime($data)
    {
        $second = $data % 60;
        $data = $data / 60;
        $minute = $data % 60;
        $data = $data / 60;
        $hour = $data % 24;
        $data = $data / 24;
        $day = $data % 31 + 1;
        $data = $data / 31;
        $month = $data % 12 + 1;
        $data = $data / 12;
        $year = floor($data + 2000);
        $d = date("Y-m-d H:i:s", strtotime($year . '-' . $month . '-' . $day . ' ' . $hour . ':' . $minute . ':' . $second));
        return $d;
    }

    private function checkSum($p)
    {
        /* This function calculates the chksum of the packet to be sent to the time clock */
        $l = count($p);
        $chksum = 0;
        $i = $l;
        $j = 1;
        while ($i > 1) {
            $u = unpack('S', pack('C2', $p['c' . $j], $p['c' . ($j + 1)]));
            $chksum += $u[1];
            if ($chksum > self::USHRT_MAX) {
                $chksum -= self::USHRT_MAX;
            }
            $i -= 2;
            $j += 2;
        }
        if ($i) {
            $chksum = $chksum + $p['c' . strval(count($p))];
        }
        while ($chksum > self::USHRT_MAX) {
            $chksum -= self::USHRT_MAX;
        }
        if ($chksum > 0) {
            $chksum = -($chksum);
        } else {
            $chksum = abs($chksum);
        }
        $chksum -= 1;
        while ($chksum < 0) {
            $chksum += self::USHRT_MAX;
        }
        return pack('S', $chksum);
    }

    public function createHeader($command, $chksum, $session_id, $reply_id, $command_string)
    {
        $buf = pack('SSSS', $command, $chksum, $session_id, $reply_id) . $command_string;
        $buf = unpack('C' . (8 + strlen($command_string)) . 'c', $buf);
        $u = unpack('S', $this->checkSum($buf));
        if (is_array($u)) {
            $u = array_pop($u);
        }
        $chksum = $u;
        $reply_id += 1;
        if ($reply_id >= self::USHRT_MAX) {
            $reply_id -= self::USHRT_MAX;
        }
        $buf = pack('SSSS', $command, $chksum, $session_id, $reply_id);
        return $buf . $command_string;
    }

    private function checkValid($reply)
    {
        $u = unpack('H2h1/H2h2', substr($reply, 0, 8));
        $command = hexdec($u['h2'] . $u['h1']);
        if ($command == self::CMD_ACK_OK) {
            return true;
        } else {
            log_message('error', "ZKLib reply with code: {$command}");
            log_message('error', "ZK Reply is: {$reply}");
            return false;
        }
    }

    public function execCommand($command, $command_string = '', $offset_data = 8)
    {
        $chksum = 0;
        $session_id = $this->session_id;
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $reply_id = hexdec($u['h8'] . $u['h7']);
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), defined('MSG_EOR') ? MSG_EOR : 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6', substr($this->received_data, 0, 8));
            $this->session_id = hexdec($u['h6'] . $u['h5']);
            return substr($this->received_data, $offset_data);
        } catch (\Throwable $e) {
            return false;
        }
    }

    private function getSizeUser()
    {
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $command = hexdec($u['h2'] . $u['h1']);
        if ($command == self::CMD_PREPARE_DATA) {
            $u = unpack('H2h1/H2h2/H2h3/H2h4', substr($this->received_data, 8, 4));
            $size = hexdec($u['h4'] . $u['h3'] . $u['h2'] . $u['h1']);
            return $size;
        } else {
            return false;
        }
    }

    private function getSizeAttendance()
    {
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $command = hexdec($u['h2'] . $u['h1']);
        if ($command == self::CMD_PREPARE_DATA) {
            $u = unpack('H2h1/H2h2/H2h3/H2h4', substr($this->received_data, 8, 4));
            $size = hexdec($u['h4'] . $u['h3'] . $u['h2'] . $u['h1']);
            return $size;
        } else {
            return false;
        }
    }

    private function getSizeTemplate()
    {
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $command = hexdec($u['h2'] . $u['h1']);
        if ($command == self::CMD_PREPARE_DATA) {
            $u = unpack('H2h1/H2h2/H2h3/H2h4', substr($this->received_data, 8, 4));
            $size = hexdec($u['h4'] . $u['h3'] . $u['h2'] . $u['h1']);
            return $size;
        } else {
            return false;
        }
    }

    public function restartDevice()
    {
        $command = self::CMD_RESTART;
        $command_string = chr(0) . chr(0);
        return $this->execCommand($command, $command_string);
    }

    public function shutdownDevice()
    {
        $command = self::CMD_POWEROFF;
        $command_string = chr(0) . chr(0);
        return $this->execCommand($command, $command_string);
    }

    public function sleepDevice()
    {
        $command = self::CMD_SLEEP;
        $command_string = chr(0) . chr(0);
        return $this->execCommand($command, $command_string);
    }

    public function resumeDevice()
    {
        $command = self::CMD_RESUME;
        $command_string = chr(0) . chr(0);
        return $this->execCommand($command, $command_string);
    }

    public function changeSpeed($speed = 0)
    {
        if ($speed != 0) {
            $speed = 1;
        }
        $command = self::CMD_CHANGE_SPEED;
        $byte = chr($speed);
        $command_string = $byte;
        return $this->execCommand($command, $command_string);
    }

    public function writeLCD($rank, $text)
    {
        $command = self::CMD_WRITE_LCD;
        $byte1 = chr((int) ($rank % 256));
        $byte2 = chr((int) ($rank >> 8));
        $byte3 = chr(0);
        $command_string = $byte1 . $byte2 . $byte3 . ' ' . $text;
        return $this->execCommand($command, $command_string);
    }

    public function clearLCD()
    {
        $command = self::CMD_CLEAR_LCD;
        return $this->execCommand($command);
    }

    public function testVoice()
    {
        $command = self::CMD_TESTVOICE;
        $command_string = chr(0) . chr(0);
        return $this->execCommand($command, $command_string);
    }

    public function getVersion()
    {
        $command = self::CMD_GET_VERSION;
        return $this->execCommand($command);
    }

    public function getOSVersion($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~OS';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setOSVersion($osVersion)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = '~OS=' . $osVersion;
        return $this->execCommand($command, $command_string);
    }

    public function getPlatform($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~Platform';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setPlatform($patform)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~Platform=' . $patform;
        return $this->execCommand($command, $command_string);
    }

    public function getFirmwareVersion($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~ZKFPVersion';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setFirmwareVersion($firmwareVersion)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = '~ZKFPVersion=' . $firmwareVersion;
        return $this->execCommand($command, $command_string);
    }

    public function getWorkCode($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = 'WorkCode';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setWorkCode($workCode)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = 'WorkCode=' . $workCode;
        return $this->execCommand($command, $command_string);
    }

    public function getSSR($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~SSR';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setSSR($ssr)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = '~SSR=' . $ssr;
        return $this->execCommand($command, $command_string);
    }

    public function getPinWidth($net = true)
    {
        $command = self::CMD_GET_PINWIDTH;
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~PIN2Width';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setPinWidth($pinWidth)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = '~PIN2Width=' . $pinWidth;
        return $this->execCommand($command, $command_string);
    }

    public function getFaceFunctionOn($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = 'FaceFunOn';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setFaceFunctionOn($faceFunctionOn)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = 'FaceFunOn=' . $faceFunctionOn;
        return $this->execCommand($command, $command_string);
    }

    public function getSerialNumber($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~SerialNumber';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setSerialNumber($serialNumber)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = '~SerialNumber=' . $serialNumber;
        return $this->execCommand($command, $command_string);
    }

    public function getDeviceName($net = true)
    {
        $command = self::CMD_OPTIONS_RRQ;
        $command_string = '~DeviceName';
        $return = $this->execCommand($command, $command_string);
        if ($net) {
            $arr = explode("=", $return, 2);
            return $arr[1];
        } else {
            return $return;
        }
    }

    public function setDeviceName($deviceName)
    {
        $command = self::CMD_OPTIONS_WRQ;
        $command_string = '~DeviceName=' . $deviceName;
        return $this->execCommand($command, $command_string);
    }

    public function getTime()
    {
        // resolution = 1 minute
        $command = self::CMD_GET_TIME;
        return $this->decodeTime(hexdec($this->reverseHex(bin2hex($this->execCommand($command)))));
    }

    public function setTime($t)
    {
        // resolution = 1 second
        $command = self::CMD_SET_TIME;
        $command_string = pack('I', $this->encodeTime($t));
        return $this->execCommand($command, $command_string);
    }

    public function enableDevice()
    {
        $command = self::CMD_ENABLEDEVICE;
        return $this->execCommand($command);
    }

    public function disableDevice()
    {
        $command = self::CMD_DISABLEDEVICE;
        $command_string = chr(0) . chr(0);
        return $this->execCommand($command, $command_string);
    }

    public function enableClock($mode = 0)
    {
        $command = self::CMD_ENABLE_CLOCK;
        $command_string = chr($mode);
        return $this->execCommand($command, $command_string);
    }

    public function getSelectedUser($uid, $finger)
    {
        $command = self::CMD_USERTEMP_RRQ;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2 . chr($finger);
        return $this->execCommand($command, $command_string);
    }

    public function getUser()
    {
        $command = self::CMD_USERTEMP_RRQ;
        $command_string = chr(5);
        $chksum = 0;
        $session_id = $this->session_id;
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $reply_id = hexdec($u['h8'] . $u['h7']);
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6', substr($this->received_data, 0, 8));
            $bytes = $this->getSizeUser();
            if ($bytes) {
                while ($bytes > 0) {
                    socket_recvfrom($this->socket, $received_data, 1032, 0, $this->ip, $this->port);
                    array_push($this->user_data, $received_data);
                    $bytes -= 1024;
                }
                $this->session_id = hexdec($u['h6'] . $u['h5']);
                socket_recvfrom($this->socket, $received_data, 1024, 0, $this->ip, $this->port);
            }
            $users = array();
            if (count($this->user_data) > 0) {
                for ($x = 0; $x < count($this->user_data); $x++) {
                    if ($x > 0) {
                        $this->user_data[$x] = substr($this->user_data[$x], 8);
                    }
                }
                $user_data = implode('', $this->user_data);
                $user_data = substr($user_data, 11);
                while (strlen($user_data) > 72) {
                    $u = unpack('H144', substr($user_data, 0, 72));
                    $u1 = hexdec(substr($u[1], 2, 2));
                    $u2 = hexdec(substr($u[1], 4, 2));
                    $uid = $u1 + ($u2 * 256); // 2 byte
                    $role = hexdec(substr($u[1], 6, 2)) . ' '; // 1 byte
                    $password = hex2bin(substr($u[1], 8, 16)) . ' '; // 8 byte
                    $name = hex2bin(substr($u[1], 24, 74)) . ' '; // 37 byte
                    $userid = hex2bin(substr($u[1], 98, 72)) . ' '; // 36 byte
                    $passwordArr = explode(chr(0), $password, 2); // explode to array
                    $password = $passwordArr[0]; // get password
                    $useridArr = explode(chr(0), $userid, 2); // explode to array
                    $userid = $useridArr[0]; // get user ID
                    $nameArr = explode(chr(0), $name, 3); // explode to array
                    $name = $nameArr[0]; // get name
                    if ($name == "") {
                        $name = $uid;
                    }
                    $users[$uid] = array($userid, $name, intval($role), $password);
                    $user_data = substr($user_data, 72);
                }
            }
            return $users;
        } catch (\Throwable $e) {
            return false;
        }
    }

    public function getUserTemplateAll($uid)
    {
        $template = array();
        $j = 0;
        for ($i = 5; $i < 10; $i++, $j++) {
            $template[$j] = $this->getUserTemplate($uid, $i);
        }
        for ($i = 4; $i >= 0; $i--, $j++) {
            $template[$j] = $this->getUserTemplate($uid, $i);
        }
        return $template;
    }

    public function getUserTemplate($uid, $finger)
    {
        $template_data = '';
        $this->user_data = array();
        $command = self::CMD_USERTEMP_RRQ;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2 . chr($finger);
        $chksum = 0;
        $session_id = $this->session_id;
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $reply_id = hexdec($u['h8'] . $u['h7']);
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6', substr($this->received_data, 0, 8));
            $bytes = $this->getSizeTemplate();
            if ($bytes) {
                while ($bytes > 0) {
                    socket_recvfrom($this->socket, $received_data, 1032, 0, $this->ip, $this->port);
                    array_push($this->user_data, $received_data);
                    $bytes -= 1024;
                }
                $this->session_id = hexdec($u['h6'] . $u['h5']);
                socket_recvfrom($this->socket, $received_data, 1024, 0, $this->ip, $this->port);
            }
            $template_data = array();
            if (count($this->user_data) > 0) {
                for ($x = 0; $x < count($this->user_data); $x++) {
                    if ($x == 0) {
                        $this->user_data[$x] = substr($this->user_data[$x], 8);
                    } else {
                        $this->user_data[$x] = substr($this->user_data[$x], 8);
                    }
                }
                $user_data = implode('', $this->user_data);
                $template_size = strlen($user_data) + 6;
                $prefix = chr($template_size % 256) . chr(round($template_size / 256)) . $byte1 . $byte2 . chr($finger) . chr(1);
                $user_data = $prefix . $user_data;
                if (strlen($user_data) > 6) {
                    $valid = 1;
                    $template_data = array($template_size, $uid, $finger, $valid, $user_data);
                }
            }
            return $template_data;
        } catch (\Throwable $e) {
            return false;
        }
    }

    public function getUserData()
    {
        $uid = 1;
        $command = self::CMD_USERTEMP_RRQ;
        $command_string = chr(5);
        $chksum = 0;
        $session_id = $this->session_id;
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $reply_id = hexdec($u['h8'] . $u['h7']);
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6', substr($this->received_data, 0, 8));
            $bytes = $this->getSizeUser();
            if ($bytes) {
                while ($bytes > 0) {
                    socket_recvfrom($this->socket, $received_data, 1032, 0, $this->ip, $this->port);
                    array_push($this->user_data, $received_data);
                    $bytes -= 1024;
                }
                $this->session_id = hexdec($u['h6'] . $u['h5']);
                socket_recvfrom($this->socket, $received_data, 1024, 0, $this->ip, $this->port);
            }
            $users = array();
            $retdata = "";
            if (count($this->user_data) > 0) {
                for ($x = 0; $x < count($this->user_data); $x++) {
                    if ($x > 0) {
                        $this->user_data[$x] = substr($this->user_data[$x], 8);
                    }
                    if ($x > 0) {
                        $retdata .= substr($this->user_data[$x], 0);
                    } else {
                        $retdata .= substr($this->user_data[$x], 12);
                    }
                }
            }
            return $retdata;
        } catch (\Throwable $e) {
            return false;
        }
    }

    public function setUser($uid, $userid, $name, $password, $role)
    {
        $uid = (int) $uid;
        $role = (int) $role;
        if ($uid > self::USHRT_MAX) {
            return false;
        }
        if ($role > 255) {
            $role = 255;
        }
        $name = substr($name, 0, 28);
        $command = self::CMD_USER_WRQ;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2 . chr($role) . str_pad($password, 8, chr(0)) . str_pad($name, 28, chr(0)) . str_pad(chr(1), 9, chr(0)) . str_pad($userid, 8, chr(0)) . str_repeat(chr(0), 16);
        return $this->execCommand($command, $command_string);
    }

    public function setUserTemplate($data)
    {
        $command = self::CMD_USERTEMP_WRQ;
        $command_string = $data;
        // $length = ord(substr($command_string, 0, 1)) + ord(substr($command_string, 1, 1)) * 256;
        return $this->execCommand($command, $command_string);
        /*
    $chksum = 0;
    $session_id = $this->session_id;
    $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
    $reply_id = hexdec($u['h8'] . $u['h7']);
    $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
    socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
    try {
    $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6', substr($this->received_data, 0, 8));
    $this->session_id = hexdec($u['h6'] . $u['h5']);
    return substr($this->received_data, 8);
    } catch (ErrorException $e) {
    return FALSE;
    } catch (exception $e) {
    return FALSE;
    }
     */
    }

    public function clearData()
    {
        $command = self::CMD_CLEAR_DATA;
        return $this->execCommand($command);
    }

    public function clearUser()
    {
        $command = self::CMD_CLEAR_DATA;
        return $this->execCommand($command);
    }

    public function deleteUser($uid)
    {
        $command = self::CMD_DELETE_USER;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2;
        return $this->execCommand($command, $command_string);
    }

    public function deleteUserTemp($uid, $finger)
    {
        $command = self::CMD_DELETE_USERTEMP;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2 . chr($finger);
        return $this->execCommand($command, $command_string);
    }

    public function clearAdmin()
    {
        $command = self::CMD_CLEAR_ADMIN;
        return $this->execCommand($command);
    }

    public function testUserTemplate($uid, $finger)
    {
        $command = self::CMD_TEST_TEMP;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2 . chr($finger);
        $u = unpack('H2h1/H2h2', $this->execCommand($command, $command_string));
        $ret = hexdec($u['h2'] . $u['h1']);
        return ($ret == self::CMD_ACK_OK) ? 1 : 0;
    }

    public function startVerify($uid)
    {
        $command = self::CMD_STARTVERIFY;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2;
        return $this->execCommand($command, $command_string);
    }

    public function startEnroll($uid, $finger)
    {
        $command = self::CMD_STARTENROLL;
        $byte1 = chr((int) ($uid % 256));
        $byte2 = chr((int) ($uid >> 8));
        $command_string = $byte1 . $byte2 . chr($finger);
        return $this->execCommand($command, $command_string);
    }

    public function cancelCapture()
    {
        $command = self::CMD_CANCELCAPTURE;
        return $this->execCommand($command);
    }

    public function getAttendance()
    {
        $command = self::CMD_ATTLOG_RRQ;
        $command_string = '';
        $chksum = 0;
        $session_id = $this->session_id;
        $u = unpack('H2h1/H2h2/H2h3/H2h4/H2h5/H2h6/H2h7/H2h8', substr($this->received_data, 0, 8));
        $reply_id = hexdec($u['h8'] . $u['h7']);
        $buf = $this->createHeader($command, $chksum, $session_id, $reply_id, $command_string);
        socket_sendto($this->socket, $buf, strlen($buf), 0, $this->ip, $this->port);
        try {
            socket_recvfrom($this->socket, $this->received_data, 1024, 0, $this->ip, $this->port);
            $bytes = $this->getSizeAttendance();
            if ($bytes) {
                while ($bytes > 0) {
                    socket_recvfrom($this->socket, $received_data, 1032, 0, $this->ip, $this->port);
                    array_push($this->attendance_data, $received_data);
                    $bytes -= 1024;
                }
                $this->session_id = hexdec($u['h6'] . $u['h5']);
                socket_recvfrom($this->socket, $received_data, 1024, 0, $this->ip, $this->port);
            }
            $attendance = array();
            if (count($this->attendance_data) > 0) {
                for ($x = 0; $x < count($this->attendance_data); $x++) {
                    if ($x > 0) {
                        $this->attendance_data[$x] = substr($this->attendance_data[$x], 8);
                    }
                }
                $attendance_data = implode('', $this->attendance_data);
                $attendance_data = substr($attendance_data, 10);
                while (strlen($attendance_data) > 40) {
                    $u = unpack('H78', substr($attendance_data, 0, 39));
                    $u1 = hexdec(substr($u[1], 4, 2));
                    $u2 = hexdec(substr($u[1], 6, 2));
                    $uid = $u1 + ($u2 * 256);
                    $id = str_replace("\0", '', hex2bin(substr($u[1], 8, 16)));
                    $state = hexdec(substr($u[1], 56, 2));
                    $timestamp = $this->decodeTime(hexdec($this->reverseHex(substr($u[1], 58, 8))));
                    $type = hexdec($this->reverseHex(substr($u[1], 66, 2)));
                    array_push($attendance, array($uid, $id, $state, $timestamp, $type));
                    $attendance_data = substr($attendance_data, 40);
                }
            }
            return $attendance;
        } catch (\Throwable $e) {
            return false;
        }
    }

    public function clearAttendance()
    {
        $command = self::CMD_CLEAR_ATTLOG;
        return $this->execCommand($command);
    }
}
