<?php
class Multiupload
{
    private $_types = [
        'pdf' => [
            'mime' => 'application/pdf',
            'ext' => ['pdf'],
        ],
        'image' => [
            'mime' => 'image/*',
            'ext' => ['jpg', 'png', 'jpeg', 'gif'],
        ],
    ];
    private $_default_options = [
        'size' => 0, // KBytes
        'type' => '*', // image, pdf, *
        'path' => './',
        'sensitize' => true,
    ];
    private $_phpFileUploadErrors = [
        1 => 'The uploaded file exceeds the upload_max_filesize directive in php.ini',
        2 => 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form',
        3 => 'The uploaded file was only partially uploaded',
        4 => 'No file was uploaded',
        6 => 'Missing a temporary folder',
        7 => 'Failed to write file to disk.',
        8 => 'A PHP extension stopped the file upload.',
    ];

    public function __construct()
    {
        if (defined('WRITEPATH')) {
            $this->_default_options['path'] = WRITEPATH;
        }
    }

    public function upload($field, $options = [])
    {
        $options = array_merge($this->_default_options, $options);
        if (!isset($_FILES[$field])) {
            return [
                'status' => 'error',
                'code' => 'not-submitted',
                'message' => 'Field not included in the form submission',
            ];
        }
        $files = [];
        if (is_array($_FILES[$field]['name'])) {
            foreach ($_FILES[$field]['name'] as $key => $file) {
                $file = [
                    'name' => $_FILES[$field]['name'][$key],
                    'type' => $_FILES[$field]['type'][$key],
                    'tmp_name' => $_FILES[$field]['tmp_name'][$key],
                    'error' => $_FILES[$field]['error'][$key],
                    'size' => $_FILES[$field]['size'][$key],
                ];
                $res = $this->_validate($file, $options);
                if ($res['status'] == 'success') {
                    $files[] = $this->_move($file, $options);
                } else {
                    return $res;
                }
            }
            return [
                'status' => 'success',
                'file' => $files,
            ];
        } else {
            $res = $this->_validate($_FILES[$field], $options);
            if ($res['status'] == 'success') {
                return ['status' => 'success', 'file' => $this->_move($_FILES[$field], $options)];
            }
            return $res;
        }
    }

    private function _validate($file, $option)
    {
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception(sprintf('Error (%s): %s', $file['name'], ($this->_phpFileUploadErrors[$file['error']] ? $this->_phpFileUploadErrors[$file['error']] : 'Unknown Upload Error')));
        }
        if (!is_uploaded_file($file['tmp_name'])) {
            throw new Exception(sprintf('Error: %s (%s) is not a valid upload file', $file['NAME'], $file['tmp_name']));
        }
        if ($option['size'] !== 0) {
            if ($file['size'] > ($option['size'] * 1024)) {
                return [
                    'status' => 'error',
                    'code' => 'file-size-exceeds',
                    'message' => sprintf('File "%s" is larger than %s', $file['name'], $this->readableBytes($option['size'] * 1024)),
                ];
            }
        }
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        if ($option['type'] != '*') {
            if (!is_array($option['type'])) {
                $option['type'] = [$option['type']];
            }
            $type_matched = false;
            $ext_matched = false;
            $last_type = end($option['type']);
            reset($option['type']);
            $file['type'] = explode('/', $file['type']);
            foreach ($option['type'] as $type) {
                if (!isset($this->_types[$type])) {
                    throw new Exception("Error: Invalid type $type specified in options");
                }
                $type_meta = $this->_types[$type];
                if (!$ext_matched) {
                    $ext_matched = in_array(strtolower($ext), $type_meta['ext']);
                }
                $type_meta['mime'] = explode('/', $type_meta['mime']);
                if (!$type_matched && $type_meta['mime'][0] == $file['type'][0] && ($type_meta['mime'][1] == '*' || $type_meta['mime'][1] == $file['type'][1])) {
                    $type_matched = true;
                }
                if ($type_matched && $ext_matched) {
                    break;
                } elseif ($type == $last_type) {
                    return [
                        'status' => 'error',
                        'code' => 'invalid-type',
                        'message' => sprintf('File "%s" is not allowed, only %s types are allowed.', $file['name'], implode(', ', $option['type'])),
                    ];
                }
            }
        }
        return ['status' => 'success'];
    }

    private function _move($file, $option)
    {
        $file_name = trim(pathinfo($file['name'], PATHINFO_FILENAME));
        $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        if ($option['sensitize']) {
            $file_name = preg_replace('#\W+#is', '_', $file_name);
        }
        $file_path = $option['path'] . "{$file_name}.{$file_ext}";
        $i = 1;
        while (file_exists($file_path)) {
            $newfile_name = $file_name . "_{$i}";
            $file_path = $option['path'] . "{$newfile_name}.{$file_ext}";
            $i++;
        }
        if (move_uploaded_file($file['tmp_name'], $file_path)) {
            if(@$option['return_url']) {
                $fc_path = str_replace(['/', '\\'], '/', FCPATH);
                $file_path = preg_replace('#[\\\/]+#', '/', $file_path);
                $file_path = str_replace($fc_path, './', $file_path);
                return [
                    'name' => $file['name'],
                    'path' => $file_path,
                ];
            } else {
                return [
                    'name' => $file['name'],
                    'path' => str_replace('\\', '/', defined('WRITEPATH') ? str_replace(WRITEPATH, '', $file_path) : $file_path),
                ];
            }
        }
        throw new Exception(sprintf('Error: Unable to move uploaded file %s to %s', $file['name'], $file_path));
    }

    public function readableBytes($bytes)
    {
        $i = floor(log($bytes) / log(1024));

        $sizes = array('B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');

        return sprintf('%.02F', $bytes / pow(1024, $i)) * 1 . ' ' . $sizes[$i];
    }
}
